package handlers

import (
	"edge-api/internal/database"
	"edge-api/internal/models"
	"edge-api/internal/services"
	"net/http"
	"time"

	"github.com/gin-gonic/gin"
)

func Login(jwtSecret string, jwtExpiration time.Duration) gin.HandlerFunc {
	return func(c *gin.Context) {

		var loginDetails models.LoginDetails

		if err := c.ShouldBindJSON(&loginDetails); err != nil {
			c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid request"})
			return
		}

		user := models.User{}
		err := database.DB.Get(&user, "select * from users where username = $1", loginDetails.UserName)
		if err != nil || !services.CheckPasswordHash(loginDetails.Password, user.PasswordHash) {
			c.JSON(http.StatusUnauthorized, gin.H{"error": "Invalid credentials"})
			return
		}

		apiKey, err := services.GenerateJWT(jwtSecret, jwtExpiration, user)
		if err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Could not generate API key"})
			return
		}

		c.SetCookie("apiKey", apiKey, int(jwtExpiration.Seconds()), "/", "", false, false)
		c.JSON(http.StatusOK, gin.H{"apiKey": apiKey})
	}
}
func Logout() gin.HandlerFunc {
	return func(c *gin.Context) {
		// Clear the httpOnly cookie
		c.SetCookie(
			"apiKey",
			"",
			-1,
			"/",
			"",
			false,
			false,
		)

		c.JSON(http.StatusOK, gin.H{
			"message": "Logged out successfully",
		})
	}
}
func Verify(jwtSecret string) gin.HandlerFunc {
	return func(c *gin.Context) {
		var apiKey string

		// Try to get token from cookie first
		apiKey, err := c.Cookie("apiKey")
		if err != nil {
			// If no cookie, try Authorization header
			authHeader := c.GetHeader("Authorization")
			if authHeader == "" {
				c.JSON(http.StatusUnauthorized, gin.H{"valid": false, "error": "No authentication token found"})
				return
			}

			// Extract token from "Bearer <token>"
			const bearerPrefix = "Bearer "
			if len(authHeader) > len(bearerPrefix) && authHeader[:len(bearerPrefix)] == bearerPrefix {
				apiKey = authHeader[len(bearerPrefix):]
			} else {
				c.JSON(http.StatusUnauthorized, gin.H{"valid": false, "error": "Invalid authorization header format"})
				return
			}
		}

		// Verify and parse the JWT
		claims, err := services.VerifyJWT(jwtSecret, apiKey)
		if err != nil {
			c.JSON(http.StatusUnauthorized, gin.H{"valid": false, "error": "Invalid or expired token"})
			return
		}

		// Token is valid
		c.JSON(http.StatusOK, gin.H{
			"valid": true,
			"user": gin.H{
				"id":    claims.UserID,
				"username": claims.UserName,
			},
		})
	}
}
