package handlers

import (
	"edge-api/internal/database"
	"edge-api/internal/models"
	"edge-api/internal/services"
	"log"
	"net/http"
	"strings"

	"github.com/gin-gonic/gin"
	"github.com/google/uuid"
	"modernc.org/sqlite"
)

func DeviceList(c *gin.Context) {
	network_id := c.Param("network_id")
	devices := []models.Device{}

	// Query network
	var network models.Network
	if err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", network_id); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Network not found"})
		return
	}

	// Query devices
	err := database.DB.Select(&devices, "SELECT devices.*, aps.name AS currentAP_name FROM devices LEFT JOIN aps ON devices.currentAP = aps.id WHERE devices.network=?", network_id)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to fetch devices"})
		log.Println("could not fetch devices", err)
		return
	}

	c.JSON(http.StatusOK, devices)
}

func DeviceListNoNetwork(c *gin.Context) {
	devices := []models.Device{}

	// Query devices
	err := database.DB.Select(&devices, "SELECT devices.*,aps.name AS currentAP_name FROM devices LEFT JOIN aps ON devices.currentAP = aps.id")
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to fetch devices"})
		return
	}

	c.JSON(http.StatusOK, devices)
}

func DeviceDetails(c *gin.Context) {
	network_id := c.Param("network_id")
	id := c.Param("device_id")

	// Query network
	var network models.Network
	if err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", network_id); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Network not found"})
		return
	}

	var device models.Device
	err := database.DB.Get(&device, "SELECT * FROM devices WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Device not found"})
		return
	}

	c.JSON(http.StatusOK, device)
}

func DeviceDetailsNoNetwork(c *gin.Context) {
	id := c.Param("device_id")

	var device models.Device
	err := database.DB.Get(&device, "SELECT * FROM devices WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Device not found"})
		return
	}

	c.JSON(http.StatusOK, device)
}

func DeviceAdd(c *gin.Context) {
	//	user_email := c.MustGet("email")
	network_id := c.Param("network_id")

	networkIdUUID, err := uuid.Parse(network_id)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "Invalid network ID"})
		return
	}

	var new_device models.DeviceNew

	if err := c.ShouldBindJSON(&new_device); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	// Query network
	var network models.Network
	if err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", network_id); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Network not found"})
		return
	}

	if network.Type == models.NetworkTypeGuest {
		c.JSON(http.StatusForbidden, gin.H{"error": "Can not add a device to a guest network"})
		return
	}

	password, err := services.RandomLexicalPassword(4)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to add device"})
		return
	}

	// // Get user ID
	// var user models.User
	// if err := database.DB.Get(&user.Email, "SELECT * FROM users WHERE email=?", user_email); err != nil {
	// 	c.JSON(http.StatusInternalServerError, gin.H{"error": "User not found"})
	// }

	// Add device
	id := uuid.New()

	_, err = database.DB.Exec("INSERT INTO devices (id, name, type, password, network, mac) values ($1, $2, $3, $4, $5, $6)",
		id, new_device.Name, new_device.Type, password, network.ID, "ff:ff:ff:ff:ff:ff")
	if err != nil {
		if sqliteErr, ok := err.(*sqlite.Error); ok && strings.Contains(sqliteErr.Error(), "UNIQUE constraint failed") {
			c.JSON(http.StatusConflict, gin.H{"error": "A device with this configuration already exists."})
			return
		}

		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to add device"})
		return
	}

	if network.Type == models.NetworkTypeBusiness {
		if err := services.UpdateConfig(); err != nil {
			log.Println(err)
			return
		}
	}

	// Query created device
	var device models.Device
	err = database.DB.Get(&device, "SELECT * FROM devices WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to add device"})
		return
	}

	// Send PSK to controller
	if network.Type == models.NetworkTypeBusiness {
		if err := services.UpdatePSKForNetwork(networkIdUUID); err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to update edge controller config"})

			// Remove device
			database.DB.Exec("DELETE FROM devices WHERE id=?", id)
			return
		}
	}

	c.JSON(http.StatusOK, device)
}

func DeviceRemoveNoNetwork(c *gin.Context) {
	id := c.Param("device_id")

	// Query device
	var device models.Device
	err := database.DB.Get(&device, "SELECT * FROM devices WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to query device"})
		return
	}

	// Query network
	var network models.Network
	if err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", device.Network.String()); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Network not found"})
		return
	}

	// Delete Device from database
	result, err := database.DB.Exec("DELETE FROM devices WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Failed to delete device"})
		return
	}

	// Delete from Controller
	if network.Type == models.NetworkTypeBusiness {
		if err := services.UpdatePSKForNetwork(network.ID); err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to update edge controller config"})
			return
		}
	}

	// Check if any rows were affected
	rowsAffected, err := result.RowsAffected()
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to delete device"})
		return
	}

	if rowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{"error": "Device not found"})
		return
	}

	// Update config on edge controller
	if err := services.UpdateConfig(); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to update edge controller config"})
		log.Println(err)
		return
	}

	c.JSON(http.StatusOK, gin.H{"message": "Device deleted successfully"})
}
