package handlers

import (
	"edge-api/internal/database"
	"edge-api/internal/models"
	"edge-api/internal/services"
	"log"
	"net/http"
	"strings"

	"github.com/gin-gonic/gin"
	"github.com/google/uuid"
	"github.com/guregu/null/v5"
	"modernc.org/sqlite"
)

func NetworkList(c *gin.Context) {
	networks := []models.Network{}

	err := database.DB.Select(&networks, "SELECT * FROM networks")
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to fetch networks"})
		return
	}

	for i := range networks {
		network := &networks[i]
		network.AddClientInfo()
	}

	c.JSON(http.StatusOK, networks)
}

func NetworkDetails(c *gin.Context) {
	id := c.Param("network_id")

	var network models.Network

	err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", id)

	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Network not found"})
		return
	}
	err = network.AddClientInfo()
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "could not poulate client info"})
	}

	c.JSON(http.StatusOK, network)
}

func NetworkCreate(c *gin.Context) {
	var new_network models.NetworkNew

	if err := c.ShouldBindJSON(&new_network); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	// Get new VLAN
	var maxVLAN int
	err := database.DB.Get(&maxVLAN, "SELECT COALESCE(MAX(vlan), 10) FROM networks")
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to create network"})
		return
	}

	/* Assign VLANs starting with ID 100 to leave room for service-vlans */
	if maxVLAN < 10 {
		maxVLAN = 100
	}

	// Add network
	id := uuid.New()
	password := null.String{}
	if new_network.Type == models.NetworkTypeGuest {
		rp, err := services.RandomLexicalPassword(4)
		if err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to create network"})
			return
		}
		password = null.StringFrom(rp)
	}
	_, err = database.DB.Exec("INSERT INTO networks (id, name, type, vlan, password) values ($1, $2, $3, $4, $5)",
		id, new_network.Name, new_network.Type, maxVLAN+1, password)
	if err != nil {
		if sqliteErr, ok := err.(*sqlite.Error); ok && strings.Contains(sqliteErr.Error(), "UNIQUE constraint failed") {
			c.JSON(http.StatusConflict, gin.H{"error": "A network with this configuration already exists."})
			return
		}

		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to create network"})
		return
	}

	// Query created network
	var network models.Network
	err = database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to create network"})
		return
	}

	// Update config on edge controller
	if err := services.UpdateConfig(); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to update edge controller config"})
		log.Println(err)
		return
	}

	c.JSON(http.StatusOK, network)
}

func NetworkPassword(c *gin.Context) {
	id := c.Param("network_id")

	var network models.Network

	err := database.DB.Get(&network, "SELECT * FROM networks WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Network not found"})
		return
	}

	if network.Type != models.NetworkTypeGuest {
		c.JSON(http.StatusMethodNotAllowed, gin.H{"error": "Network is not a guest network"})
	}

	c.JSON(http.StatusOK, gin.H{"password": network.Password})
}

func NetworkDelete(c *gin.Context) {
	id := c.Param("network_id")

	result, err := database.DB.Exec("PRAGMA foreign_keys = ON; DELETE FROM networks WHERE id=?", id)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "Failed to delete network"})
		return
	}

	// Check if any rows were affected
	rowsAffected, err := result.RowsAffected()
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to delete network"})
		return
	}

	if rowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{"error": "Network not found"})
		return
	}

	// Update config on edge controller
	if err := services.UpdateConfig(); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": "Failed to update edge controller config"})
		log.Println(err)
		return
	}

	c.JSON(http.StatusOK, gin.H{"message": "Network deleted successfully"})
}
