package handlers

import (
	"encoding/json"
	"fmt"
	"github.com/gin-gonic/gin"
	"io"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"
)

type RotateRequest struct {
	PageIndex int `json:"pageIndex" binding:"required"`
	Rotation  int `json:"rotation" binding:"required"` // 90, 180, 270 degrees
}

type RotateResponse struct {
	Message   string `json:"message"`
	PageIndex int    `json:"pageIndex"`
	Rotation  int    `json:"rotation"`
	Success   bool   `json:"success"`
}
type Project struct {
	Name          string         `json:"name"`
	AccessPoints  []AccessPoint  `json:"accessPoints"`
	CurrentPage   int            `json:"currentPage"`
	Timestamp     int64          `json:"timestamp"`
	PDFFile       *PDFFile       `json:"pdfFile,omitempty"`
	PageRotation  int            `json:"pageRotation"`
	ViewportState *ViewportState `json:"viewportState,omitempty"`
}

// Viewport state for preserving zoom and pan
type ViewportState struct {
	ZoomLevel     float64                 `json:"zoomLevel"`
	PanOffset     PanOffset               `json:"panOffset"`
	PageViewports map[string]PageViewport `json:"pageViewports,omitempty"` // per-page viewport states
}
type PanOffset struct {
	X float64 `json:"x"`
	Y float64 `json:"y"`
}

type PageViewport struct {
	ZoomLevel float64   `json:"zoomLevel"`
	PanOffset PanOffset `json:"panOffset"`
}
type DeviceDetail struct {
	MacAddress       string      `json:"macAddress"`
	Bandwidth        string      `json:"bandwidth"`
	SignalStrength   json.Number `json:"signalStrength"`
	ConnectedDevices int         `json:"connectedDevices"`
	LastSeen         time.Time   `json:"lastSeen"`
	Status           string      `json:"status"`
}

type AccessPoint struct {
	ID           string        `json:"id"`
	Type         string        `json:"type"`
	X            float64       `json:"x"`
	Y            float64       `json:"y"`
	Page         int           `json:"page"`
	Floor        string        `json:"floor"`
	Name         string        `json:"name"`
	DeviceDetail *DeviceDetail `json:"deviceDetail,omitempty"`
}

type PDFFile struct {
	Name         string `json:"name"`
	Size         int64  `json:"size"`
	OriginalName string `json:"originalName"`
	UploadedPath string `json:"uploadedPath"`
}

type UploadResponse struct {
	Message      string `json:"message"`
	FilePath     string `json:"filePath"`
	OriginalName string `json:"originalName"`
	Filename     string `json:"filename"`
	Size         int64  `json:"size"`
}

type ProjectResponse struct {
	Message string  `json:"message"`
	Project Project `json:"project"`
}

type ErrorResponse struct {
	Error   string `json:"error"`
	Details string `json:"details,omitempty"`
}

const (
	DataDir      = "./data"
	UploadsDir   = "./uploads"
	ProjectsFile = "./data/projects.json"
)

func UploadPDF(c *gin.Context) {
	// Parse multipart form
	file, header, err := c.Request.FormFile("pdf")
	if err != nil {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "No PDF file uploaded",
		})
		return
	}
	defer file.Close()

	// Validate file type
	if header.Header.Get("Content-Type") != "application/pdf" {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "Only PDF files are allowed",
		})
		return
	}

	// Check file size (50MB limit)
	const maxFileSize = 50 * 1024 * 1024 // 50MB
	if header.Size > maxFileSize {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "File size exceeds 50MB limit",
		})
		return
	}

	// Get original name from form or header
	originalName := c.PostForm("originalName")
	if originalName == "" {
		originalName = header.Filename
	}

	// Generate unique filename
	timestamp := time.Now().Unix()
	ext := filepath.Ext(originalName)
	baseName := strings.TrimSuffix(originalName, ext)
	filename := fmt.Sprintf("%s-%d%s", baseName, timestamp, ext)

	// Create full file path
	filePath := filepath.Join(UploadsDir, filename)

	// Create the file
	dst, err := os.Create(filePath)
	if err != nil {
		log.Printf("Error creating file: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to create file",
			Details: err.Error(),
		})
		return
	}
	defer dst.Close()

	// Copy uploaded file to destination
	if _, err := io.Copy(dst, file); err != nil {
		log.Printf("Error copying file: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to save file",
			Details: err.Error(),
		})
		return
	}

	// Return success response
	uploadPath := "/uploads/" + filename

	log.Printf("📤 PDF uploaded: %s -> %s (size: %d bytes)", originalName, filename, header.Size)

	c.JSON(http.StatusOK, UploadResponse{
		Message:      "PDF uploaded successfully",
		FilePath:     uploadPath,
		OriginalName: originalName,
		Filename:     filename,
		Size:         header.Size,
	})
}

// Save project
func SaveProject(c *gin.Context) {
	var project Project
	if err := c.ShouldBindJSON(&project); err != nil {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error:   "Invalid project data",
			Details: err.Error(),
		})
		return
	}

	// Validate required fields
	if strings.TrimSpace(project.Name) == "" {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "Project name is required",
		})
		return
	}

	if project.PDFFile == nil || project.PDFFile.UploadedPath == "" {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "PDF file is required for project",
		})
		return
	}

	// Set timestamp if not provided
	if project.Timestamp == 0 {
		project.Timestamp = time.Now().Unix()
	}

	// Load existing projects
	projects, err := LoadProjects()
	if err != nil {
		log.Printf("Error loading projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to load existing projects",
			Details: err.Error(),
		})
		return
	}

	// Check if project exists and update or add
	updated := false
	for i, p := range projects {
		if p.Name == strings.TrimSpace(project.Name) {
			projects[i] = project
			updated = true
			log.Printf(" Updated existing project: %s", project.Name)
			break
		}
	}

	if !updated {
		projects = append(projects, project)
		log.Printf("Created new project: %s", project.Name)
	}

	// Save projects to file
	if err := SaveProjects(projects); err != nil {
		log.Printf("Error saving projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to save project",
			Details: err.Error(),
		})
		return
	}

	message := "Project created successfully"
	if updated {
		message = "Project updated successfully"
	}

	c.JSON(http.StatusOK, ProjectResponse{
		Message: message,
		Project: project,
	})
}

// Get all projects
func GetProjects(c *gin.Context) {
	projects, err := LoadProjects()
	if err != nil {
		log.Printf("Error loading projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to load projects",
			Details: err.Error(),
		})
		return
	}

	c.JSON(http.StatusOK, projects)
}

// Delete project
func DeleteProject(c *gin.Context) {
	projectName := c.Param("name")
	if projectName == "" {
		c.JSON(http.StatusBadRequest, ErrorResponse{
			Error: "Project name is required",
		})
		return
	}

	// Load existing projects
	projects, err := LoadProjects()
	if err != nil {
		log.Printf("Error loading projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to load projects",
			Details: err.Error(),
		})
		return
	}

	// Filter out the project to delete
	filteredProjects := make([]Project, 0)
	found := false
	for _, p := range projects {
		if p.Name != projectName {
			filteredProjects = append(filteredProjects, p)
		} else {
			found = true
			log.Printf(" Deleted project: %s", projectName)
		}
	}

	if !found {
		c.JSON(http.StatusNotFound, ErrorResponse{
			Error: "Project not found",
		})
		return
	}

	// Save updated projects
	if err := SaveProjects(filteredProjects); err != nil {
		log.Printf("Error saving projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to delete project",
			Details: err.Error(),
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"message": "Project deleted successfully",
	})
}

// Get first project for auto-loading
func GetFirstProject(c *gin.Context) {
	projects, err := LoadProjects()
	if err != nil {
		log.Printf("Error loading projects: %v", err)
		c.JSON(http.StatusInternalServerError, ErrorResponse{
			Error:   "Failed to load projects",
			Details: err.Error(),
		})
		return
	}

	if len(projects) == 0 {
		c.JSON(http.StatusOK, gin.H{
			"project":         nil,
			"hasAccessPoints": false,
		})
		return
	}

	// Get the most recent project (highest timestamp)
	var latestProject *Project
	for i := range projects {
		if latestProject == nil || projects[i].Timestamp > latestProject.Timestamp {
			latestProject = &projects[i]
		}
	}

	response := gin.H{
		"project":         latestProject,
		"hasAccessPoints": len(latestProject.AccessPoints) > 0,
	}

	// Add PDF URL if available
	if latestProject.PDFFile != nil && latestProject.PDFFile.UploadedPath != "" {
		response["pdfUrl"] = latestProject.PDFFile.UploadedPath
	}

	c.JSON(http.StatusOK, response)
}

// Load projects from JSON file
func LoadProjects() ([]Project, error) {
	var projects []Project

	// Check if file exists
	if _, err := os.Stat(ProjectsFile); os.IsNotExist(err) {
		// File doesn't exist, return empty slice
		return projects, nil
	}

	// Read file
	data, err := os.ReadFile(ProjectsFile)
	if err != nil {
		return nil, err
	}

	// Parse JSON
	if err := json.Unmarshal(data, &projects); err != nil {
		return nil, err
	}

	return projects, nil
}

// Save projects to JSON file
func SaveProjects(projects []Project) error {
	// Marshal to JSON with indentation
	data, err := json.MarshalIndent(projects, "", "  ")
	if err != nil {
		return err
	}

	// Write to file
	return os.WriteFile(ProjectsFile, data, 0644)
}
