package services

import (
	"context"
	"edge-api/internal/database"
	"edge-api/internal/models"
	"errors"
	"fmt"
	"log"
	"net/url"
	"strings"
	"time"

	"database/sql"

	"github.com/google/uuid"
	"github.com/guregu/null/v5"
	"github.com/jmoiron/sqlx"
	"modernc.org/sqlite"
)

// StartAPWorker runs background tasks for AP checks
func StartAPWorker(ctx context.Context) {
	log.Println("AP Worker started")
	// Check APs every 10 seconds
	ticker := time.NewTicker(10 * time.Second)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			if err := checkAPs(); err != nil {
				fmt.Println("AP check error:", err)
			}
		}
	}
}

// checkAPs checks for AP and device updates and mesh connections
func checkAPs() error {

	edgeControllerApi := getControllerApi()
	if edgeControllerApi == "" {
		log.Fatalf("EDGE_CONTROLLER_API not set")
	}
	edgeControllerApiUrl, err := url.Parse(edgeControllerApi)
	if err != nil {
		log.Fatalf("Failed to parse EDGE_CONTROLLER_API: %v", err)
	}

	api := ControllerAPI{URL: *edgeControllerApiUrl}

	nodes, err := api.getNodes()
	if err != nil {
		return err
	}

	if err := removeAPsExcept(nodes.Names()); err != nil {
		return err
	}

	if err := clearMeshConnections(); err != nil {
		return err
	}

	defaultNetworkUUID, err := GetDefaultNetworkUUID()
	if err != nil {
		fmt.Printf("Error getting default network UUID: %v\n", err)
		defaultNetworkUUID = uuid.Nil
	}

	all_metrics := make(models.NodeMetricsMap)
	for _, node := range *nodes {
		metrics, err := api.getNodeMetrics(node)
		if err != nil {
			log.Printf("Failed to get node metrics for AP %s: %v\n", node.Name, err)
			continue
		}
		all_metrics[node.Name] = metrics

		// updating the access points data
		if err := updateAP(node, metrics); err != nil {
			fmt.Printf("Failed to update AP %s: %v\n", node.Name, err)
		}

		var ap models.AP
		if err := database.DB.Get(&ap, "SELECT * FROM aps WHERE name = ?", node.Name); err != nil {
			fmt.Printf("Couldn't find AP %s in database: %v\n", node.Name, err)
			continue
		}

		// update connected client data per AP
		if metrics != nil {
			// set all devices offline after 60s timeout
			SetDevicesOfflineAfterTimeout()
			for _, clientHostapd := range metrics.ClientsHostapd {
				for mac, client := range clientHostapd.Clients {
					if client.PSK != "" {
						// Upsert (insert or update) device using PSK for authentication
						if err := UpsertDeviceFromMetrics(
							true, // Device is online
							mac,
							client.PSK,
							ap,
							client.Signal,
							client.Rate.RX,
							client.Rate.TX,
							client.Bytes.RX,
							client.Bytes.TX,
							client.Packets.RX,
							client.Packets.TX,
							defaultNetworkUUID,
						); err != nil {
							fmt.Printf("Failed to create device %s: %v\n", mac, err)
						} else {
							fmt.Printf("Processed client %s on AP %s\n", mac, node.Name)
						}
					}
				}
			}
		}
	}

	neighbors := all_metrics.FindNeighbors()
	for nodeName, neighbor := range *neighbors {
		if neighbor.Metrics.LastSeen.Duration() > 5*time.Second {
			continue
		}

		if err := InsertMeshConnection(nodeName, neighbor.Name, models.SignalStrengthGreat); err != nil {
			var sqliteErr *sqlite.Error
			if errors.As(err, &sqliteErr) {
				// Ignore unique constraint errors
			} else {
				fmt.Printf("Failed to insert mesh connection for AP %s with %s: %v\n", nodeName, neighbor.Name, err)
			}
		}
	}

	return nil
}

// updateAP updates the AP's last connected time in the database.DB
func updateAP(n *models.NodeResponse, m *models.NodeMetricsResponse) error {
	var ap models.AP
	err := database.DB.Get(&ap, "SELECT * FROM aps WHERE name = ?", n.Name)
	if err != nil {
		if errors.Is(err, sql.ErrNoRows) {
			currentTime := null.TimeFrom(time.Now())
			log.Printf("AP %s not found in database, attempting to insert...", n.Name)

			// AP not found so we need to add it
			_, insertErr := database.DB.Exec("INSERT INTO aps (id, name, firstConnectedAt, lastConnectedAt) values ($1, $2, $3, $4)",
				uuid.New(), n.Name, currentTime, currentTime)
			if insertErr != nil {
				log.Printf("Error inserting new AP %s: %v (Error Type: %T)", n.Name, insertErr, insertErr)
				return insertErr
			}
			log.Printf("Successfully inserted new AP %s into database", n.Name)
			return nil
		}
		log.Printf("Error fetching AP %s from database (not NoRows): %v (Error Type: %T)", n.Name, err, err)
		return err
	}

	_, updateErr := database.DB.Exec("UPDATE aps SET lastConnectedAt = $1 WHERE name = $2", time.Now(), ap.Name)
	if updateErr != nil {
		log.Printf("Error updating lastConnectedAt for AP %s: %v (Error Type: %T)", ap.Name, updateErr, updateErr)
		return updateErr
	}
	return nil
}

// InsertMeshConnection inserts a mesh connection between two APs
func InsertMeshConnection(ap_name string, other_ap_name string, SignalStrength models.SignalStrength) error {

	var ap models.AP
	if err := database.DB.Get(&ap, "SELECT * FROM aps WHERE name = ?", ap_name); err != nil {
		log.Printf("Error fetching AP %s from database: %v", ap_name, err)
		return err
	}

	var other_ap models.AP
	if err := database.DB.Get(&other_ap, "SELECT * FROM aps WHERE name = ?", other_ap_name); err != nil {
		log.Printf("Error fetching AP %s from database: %v", other_ap_name, err)
	}

	_, err := database.DB.Exec("INSERT INTO ap_mesh_connections (ap, otherAP, SignalStrength) VALUES ($1, $2, $3)",
		ap.ID, other_ap.ID, SignalStrength)
	if err != nil {
		return err
	}

	return nil
}

// removeAPsExcept removes APs that are not in the given list of names
func removeAPsExcept(names []string) error {

	if len(names) == 0 {
		return nil
	}

	args := make([]any, len(names))
	placeholders := make([]string, len(names))

	for i, name := range names {
		placeholders[i] = "?"
		args[i] = name
	}

	query := fmt.Sprintf("DELETE FROM aps WHERE name NOT IN (%s)", strings.Join(placeholders, ","))
	_, err := database.DB.Exec(query, args...)

	if err != nil {
		// Log the error for debugging, potentially including the query structure (but not args directly)
		fmt.Printf("Error executing delete query [%s]: %v\n", query, err)
		return fmt.Errorf("failed to execute delete statement: %w", err)
	}

	return nil
}

// clearMeshConnections removes all mesh connections from the database.DB
func clearMeshConnections() error {
	_, err := database.DB.Exec("DELETE FROM ap_mesh_connections")
	if err != nil {
		return err
	}
	return nil
}

func GetDefaultNetworkUUID() (uuid.UUID, error) {
	var network models.Network
	err := database.DB.Get(&network, "SELECT * FROM networks LIMIT 1") // **ADJUST THIS QUERY** to get your default network
	if err != nil {
		if errors.Is(err, sql.ErrNoRows) {
			return uuid.Nil, fmt.Errorf("no networks found in database to use as default")
		}
		return uuid.Nil, fmt.Errorf("failed to fetch default network: %w", err)
	}
	return network.ID, nil
}

func CheckAPOnlineStatus(ap models.AP) models.AP {
	timeout := 30 * time.Second
	isOnline := ap.LastConnectedAt.Valid && time.Since(ap.LastConnectedAt.Time) <= timeout

	ap.Online = isOnline

	return ap
}

func GetConnectedDevicesPerAP(ap models.AP, db *sqlx.DB) (models.AP, error) {
	var connectedDeviceCount int

	query := "SELECT count(*) FROM devices WHERE online = ? AND currentAP = ?"

	err := db.Get(&connectedDeviceCount, query, true, ap.ID)

	if err != nil {
		return ap, fmt.Errorf("error counting online devices for AP %s from database: %w", ap.ID, err)
	}

	ap.ConnectedDevices = connectedDeviceCount

	return ap, nil
}
