package services

import (
	"database/sql"
	"edge-api/internal/database"
	"edge-api/internal/models"
	"errors"
	"fmt"
	"log"
	"time"

	"github.com/google/uuid"
	"github.com/guregu/null/v5"
)

func SetDevicesOfflineAfterTimeout() error {
	// TODO: maybe set this as global conf variable
	timeout := 10 * time.Second
	var devices models.Device

	onlineDevices, err := devices.GetAllOnline()
	if err != nil {
		fmt.Println("could not get all online devices from db")
	} else {
		for _, onlineDevice := range onlineDevices {
			if time.Since(onlineDevice.LastConnectedAt.Time) > timeout {
				if err := onlineDevice.SetDeviceOffline(); err != nil {

					fmt.Printf("Error setting device ID %s offline: %v\n", onlineDevice.ID, err)

				} else {

					// fmt.Printf("Device %s offline.\n", onlineDevice.Name)
					return nil
				}
			}
		}

	}
	return nil
}

func UpsertDeviceFromMetrics(

	onlineStatus bool,
	mac string,
	psk string,
	ap models.AP,
	SignalStrength int64,
	RateRX int64,
	RateTX int64,
	BytesRX int64,
	BytesTX int64,
	PacketsRX int64,
	PacketsTX int64,
	networkUUID uuid.UUID,

) error {

	var device models.Device

	err := device.GetByPassword(psk)

	// if device is not in DB
	if errors.Is(err, sql.ErrNoRows) {
		return fmt.Errorf("a device tries to connect, which is not created in the DB: %w", err)
	} else if err != nil {
		return err
	}

	// Update existing device
	if device.FirstConnectedAt.IsZero() {
		log.Printf("device %v using the PSK %v has connected to the ssid %v for the first time", device.Name, device.Password, device.Network)
		device.FirstConnectedAt = null.TimeFrom(time.Now())
		_, err = database.DB.NamedExec(`UPDATE devices SET
		firstConnectedAt = :firstConnectedAt
		WHERE password = :password`, device)
		return err
	}
	device.Online = onlineStatus
	device.Auth = true
	device.Authorized = true
	device.LastConnectedAt = null.TimeFrom(time.Now())
	device.SignalStrength = null.IntFrom(int64(SignalStrength))
	device.RateRX = null.IntFrom(int64(RateRX))
	device.RateTX = null.IntFrom(int64(RateTX))
	device.BytesRX = null.IntFrom(int64(BytesRX))
	device.BytesTX = null.IntFrom(int64(BytesTX))
	device.PacketsRX = null.IntFrom(int64(PacketsRX))
	device.PacketsTX = null.IntFrom(int64(PacketsTX))
	device.CurrentAP = null.StringFrom(ap.ID.String())
	device.MAC = null.StringFrom(mac)

	_, err = database.DB.NamedExec(`UPDATE devices SET
		online = :online,
		auth = :auth,
		authorized = :authorized,
		lastConnectedAt = :lastConnectedAt,
		SignalStrength = :SignalStrength,
		currentAP = :currentAP,
		rate_rx = :rate_rx,
		rate_tx = :rate_tx,
		mac = :mac,
		bytes_rx = :bytes_rx,
		bytes_tx = :bytes_tx,
		packets_rx = :packets_rx,
		packets_tx = :packets_tx
		WHERE password = :password`, device)

	return err

}
