package main

import (
	"context"
	"edge-api/internal/database"
	"edge-api/internal/handlers"
	"edge-api/internal/middlewares"
	"edge-api/internal/models"
	"edge-api/internal/repositories"
	"edge-api/internal/services"
	"log"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"time"

	"github.com/gin-gonic/gin"
)

func getJwtSecret() string {
	jwtSecret := os.Getenv("JWT_SECRET")
	if jwtSecret != "" {
		return jwtSecret
	}

	/* Check if secret is in files in data-directory */
	secretFilePath := "/lib/uniberg/edge-api/data/jwt_secret"
	if _, err := os.Stat(secretFilePath); err == nil {
		data, err := os.ReadFile(secretFilePath)
		if err != nil {
			log.Fatal("Failed to read JWT secret from file")
		}
		return strings.Replace(string(data), "\n", "", -1)
	}

	return ""
}

func getAdminPass() string {
	adminPass := os.Getenv("ADMIN_PASS")
	if adminPass != "" {
		return adminPass
	}

	/* Check if secret is in files in data-directory */
	secretFilePath := "/lib/uniberg/edge-api/data/admin_pass"
	if _, err := os.Stat(secretFilePath); err == nil {
		data, err := os.ReadFile(secretFilePath)
		if err != nil {
			log.Fatal("Failed to read admin password from file")
		}
		return strings.Replace(string(data), "\n", "", -1)
	}

	return ""
}

func getDatabasePath() string {
	/* Check if data-dir exists */
	dataDir := "/lib/uniberg/edge-api/data"
	if _, err := os.Stat(dataDir); os.IsNotExist(err) {
		return "edge.sqlite"
	}

	return "/lib/uniberg/edge-api/data/edge.sqlite"
}

const (
	DataDir      = "./data"
	UploadsDir   = "./uploads"
	ProjectsFile = "./data/projects.json"
)

func main() {

	log.SetFlags(log.Lshortfile)

	dbPath := getDatabasePath()

	// Initialize DB
	if err := database.InitDB(getDatabasePath()); err != nil {
		log.Fatal("DB init failed:", err)
	}
	log.Printf("Database path: %s", dbPath) // ADD THIS LINE
	// Execute schemas
	database.DB.MustExec(models.UserSchema)
	database.DB.MustExec(models.NetworkSchema)
	database.DB.MustExec(models.APSchema)
	database.DB.MustExec(models.DeviceSchema)
	database.DB.MustExec(models.MeshConnectionSchema)
	database.DB.MustExec(models.ConnectedDeviceSchema)
	defer database.DB.Close()

	isDev := os.Getenv("APP_ENV") == "development"
	ginMode := gin.ReleaseMode
	if isDev {
		ginMode = gin.DebugMode
	}

	listenPort := os.Getenv("LISTEN_PORT")
	if listenPort == "" {
		listenPort = "8080"
	}

	jwtSecret := getJwtSecret()
	if jwtSecret == "" {
		log.Fatal("JWT_SECRET not available")
	}

	adminPass := getAdminPass()
	if adminPass == "" {
		log.Fatal("ADMIN_PASS not available")
	}

	adminPassHash, err := services.HashPassword(adminPass)
	if err != nil {
		log.Fatal("Failed to generate admin password hash")
	}

	// Create admin user
	repositories.InitDB(getDatabasePath(), adminPassHash)

	gin.SetMode(ginMode)

	r := gin.Default()

	jwtDuration := time.Hour * 48 // TODO: use env variable or config file

	r.Static("/uploads", UploadsDir)

	createDirectories()

	// Login route
	r.POST("/auth/login", handlers.Login(jwtSecret, jwtDuration))
	r.POST("/auth/logout", handlers.Logout())
	r.GET("/auth/verify", handlers.Verify(jwtSecret))
	// Protected routes

	r.GET("/auth/users", middlewares.Auth(jwtSecret), handlers.UserList)
	r.GET("/edge/networks", middlewares.Auth(jwtSecret), handlers.NetworkList)
	r.GET("/edge/networks/:network_id", middlewares.Auth(jwtSecret), handlers.NetworkDetails)
	r.GET("/edge/networks/:network_id/guest-password", middlewares.Auth(jwtSecret), handlers.NetworkPassword)
	r.POST("/edge/networks", middlewares.Auth(jwtSecret), handlers.NetworkCreate)
	r.DELETE("/edge/networks/:network_id", middlewares.Auth(jwtSecret), handlers.NetworkDelete)
	r.GET("/edge/networks/:network_id/devices", middlewares.Auth(jwtSecret), handlers.DeviceList)
	r.GET("/edge/networks/:network_id/devices/:device_id", middlewares.Auth(jwtSecret), handlers.DeviceDetails)
	r.POST("/edge/networks/:network_id/devices", middlewares.Auth(jwtSecret), handlers.DeviceAdd)
	r.GET("/edge/devices", middlewares.Auth(jwtSecret), handlers.DeviceListNoNetwork)
	r.GET("/edge/devices/:device_id", middlewares.Auth(jwtSecret), handlers.DeviceDetailsNoNetwork)
	r.DELETE("/edge/devices/:device_id", middlewares.Auth(jwtSecret), handlers.DeviceRemoveNoNetwork)
	r.GET("/edge/aps", middlewares.Auth(jwtSecret), handlers.APList)
	r.GET("/edge/aps/:ap_id", middlewares.Auth(jwtSecret), handlers.APDetails)

	r.GET("/projects", middlewares.Auth(jwtSecret), handlers.GetProjects)
	r.POST("/projects", middlewares.Auth(jwtSecret), handlers.SaveProject)
	r.DELETE("/projects/:name", middlewares.Auth(jwtSecret), handlers.DeleteProject)
	r.GET("/projects/first/load", middlewares.Auth(jwtSecret), handlers.GetFirstProject)
	r.POST("/projects/uploads/pdf", middlewares.Auth(jwtSecret), handlers.UploadPDF)
	// Create an HTTP server
	server := &http.Server{
		Addr:    ":8080",
		Handler: r,
	}

	// Create a cancellable context to later stop workers
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// Start the AP worker
	go services.StartAPWorker(ctx)

	// Start initial config update
	go services.StartInitialConfigSync(ctx)

	// Handle graceful shutdown
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, os.Interrupt)
	go func() {
		<-quit
		log.Println("Shutting down server...")

		// Cancel the worker context to stop them
		cancel()

		// Shutdown the Gin server
		if err := server.Shutdown(context.Background()); err != nil {
			log.Fatal("Server forced to shutdown:", err)
		}
	}()

	// Start the HTTP server
	log.Printf("Starting server on port %s", listenPort)
	if err := server.ListenAndServe(); err != nil && err != http.ErrServerClosed {
		log.Fatalf("Server failed to start: %v", err)
	}
}
func createDirectories() {
	dirs := []string{DataDir, UploadsDir}
	for _, dir := range dirs {
		if err := os.MkdirAll(dir, 0755); err != nil {
			log.Fatalf("Failed to create directory %s: %v", dir, err)
		}
	}
}
