"use client"

import React from 'react';
// Custom Hooks
import AuthCheck from "../components/AuthCheck";
import { useAuthContext } from '@/features/auth';
import { useDomReady } from '@/shared/hooks';
import { useToast } from '@/components/ToastProvider';
import { useLoader } from '@/shared/providers';
// API Client
import { api } from '@/lib/api';
// External Libraries Dependencies
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiPageHeaderSection,
  EuiTitle,
  EuiBasicTable,
  EuiBasicTableColumn,
  EuiTableSortingType,
  Criteria,
  EuiButtonIcon,
  EuiHealth,
  EuiFlexGroup,
  EuiFlexItem,
  EuiScreenReaderOnly,
  EuiSpacer,
  EuiSwitch,
  EuiPanel,
  EuiLoadingSpinner,
  EuiEmptyPrompt,
  EuiIcon,
  EuiButton,
  EuiText,
  EuiBadge,
  EuiStat,
  formatDate,
  Comparators,
} from '@elastic/eui';
// Components
import Navigation from "../components/navigation/Navigation";
import { CheckQuality } from '@/components/QualityCheck';
// Own Dependencies
import { ApiError, getErrorCode } from '@/lib/errorsCodes';
import getDict from '@/lib/dict';
import AP from '../types/AP';
import { checkResponseShape } from '@/lib/checkResponseShape';

const dict = getDict("de");

export default function Antennas() {
  // State
  const [auto, setAuto] = React.useState(true);
  const [aps, setAPs] = React.useState<AP[]>([]);
  const [itemIdToExpandedRowMap, setItemIdToExpandedRowMap] = React.useState<Record<string, React.ReactNode>>({});
  const [pageIndex, setPageIndex] = React.useState(0);
  const [pageSize, setPageSize] = React.useState(10);
  const [sortField, setSortField] = React.useState<keyof AP>('name');
  const [sortDirection, setSortDirection] = React.useState<'asc' | 'desc'>('asc');

  // Hooks
  const domReady = useDomReady();
  const { isAuthenticated } = useAuthContext();
  const { setLoading } = useLoader();
  const { addToast } = useToast();

  // Fetch Access Points
  const fetchAPs = React.useCallback(async () => {
    try {
      setLoading(true);
      const data = await api.accessPoints.getAccessPoints();
      if (checkResponseShape(data)) {
        setAPs(data);
      } else {
        addToast({
          title: 'Fetch Error',
          color: "danger",
          text: dict.apFetchToastFail,
        });
      }
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('ACCESS_POINT_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: dict.apFetchToastFail,
      });
    } finally {
      setLoading(false);
    }
  }, [addToast, setLoading]);

  // Auto-fetch effect
  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;

    fetchAPs();

    if (auto) {
      const interval = setInterval(fetchAPs, 5000);
      return () => clearInterval(interval);
    }
  }, [auto, domReady, isAuthenticated, fetchAPs]);

  // Toggle row details
  const toggleDetails = (accessPoint: AP) => {
    const itemIdToExpandedRowMapValues = { ...itemIdToExpandedRowMap };

    if (itemIdToExpandedRowMapValues[accessPoint.id]) {
      delete itemIdToExpandedRowMapValues[accessPoint.id];
    } else {
      const { name, signal, connectedDevices, mac_address, online, firstConnectedAt, lastConnectedAt } = accessPoint;

      itemIdToExpandedRowMapValues[accessPoint.id] = (
        <EuiPanel color="subdued" paddingSize="l">
          <EuiFlexGroup gutterSize="l" wrap>
            {/* Status Card */}
            <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
              <EuiPanel color="primary" hasShadow={false}>
                <EuiStat
                  title={
                    <EuiHealth color={online ? 'success' : 'danger'} style={{ fontSize: '1.5rem' }}>
                      {online ? dict.online : dict.offline}
                    </EuiHealth>
                  }
                  description="Status"
                  titleSize="s"
                  textAlign="center"
                >
                  <EuiIcon type={online ? 'check' : 'cross'} size="xl" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>

            {/* Signal Quality Card */}
            <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
              <EuiPanel color="success" hasShadow={false}>
                <EuiStat
                  title={
                    <EuiFlexGroup gutterSize="s" alignItems="center" justifyContent="center">
                      <EuiFlexItem grow={false}>
                        <span>{signal || 0}%</span>
                      </EuiFlexItem>
                      <EuiFlexItem grow={false}>
                        <CheckQuality value={signal || 0} />
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  }
                  description={dict.signal}
                  titleSize="s"
                  textAlign="center"
                >
                  <EuiIcon type="online" size="xl" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>

            {/* Connected Devices Card */}
            <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
              <EuiPanel color="accent" hasShadow={false}>
                <EuiStat
                  title={connectedDevices || '0'}
                  description={dict.connectedDevices}
                  titleSize="s"
                  textAlign="center"
                >
                  <EuiIcon type="compute" size="xl" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
          </EuiFlexGroup>

          <EuiSpacer size="m" />

          {/* Additional Details */}
          <EuiFlexGroup gutterSize="l" wrap>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiText size="s">
                  <h4>
                    <EuiIcon type="tag" /> Device Information
                  </h4>
                </EuiText>
                <EuiSpacer size="s" />
                <EuiFlexGroup direction="column" gutterSize="s">
                  <EuiFlexItem>
                    <EuiFlexGroup justifyContent="spaceBetween" gutterSize="s">
                      <EuiFlexItem grow={false}>
                        <EuiText size="s" color="subdued">
                          <strong>{dict.name}:</strong>
                        </EuiText>
                      </EuiFlexItem>
                      <EuiFlexItem>
                        <EuiText size="s" textAlign="right">
                          <EuiBadge color="hollow">{name}</EuiBadge>
                        </EuiText>
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  </EuiFlexItem>
                  <EuiFlexItem>
                    <EuiFlexGroup justifyContent="spaceBetween" gutterSize="s">
                      <EuiFlexItem grow={false}>
                        <EuiText size="s" color="subdued">
                          <strong>{dict.macAddress}:</strong>
                        </EuiText>
                      </EuiFlexItem>
                      <EuiFlexItem>
                        <EuiText size="s" textAlign="right">
                          <code>{mac_address || '-'}</code>
                        </EuiText>
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPanel>
            </EuiFlexItem>

            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiText size="s">
                  <h4>
                    <EuiIcon type="clock" /> Connection History
                  </h4>
                </EuiText>
                <EuiSpacer size="s" />
                <EuiFlexGroup direction="column" gutterSize="s">
                  <EuiFlexItem>
                    <EuiFlexGroup justifyContent="spaceBetween" gutterSize="s">
                      <EuiFlexItem grow={false}>
                        <EuiText size="s" color="subdued">
                          <strong>{dict.firstConnectedAt}:</strong>
                        </EuiText>
                      </EuiFlexItem>
                      <EuiFlexItem>
                        <EuiText size="s" textAlign="right">
                          {firstConnectedAt ? formatDate(firstConnectedAt, 'dobLong') : '-'}
                        </EuiText>
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  </EuiFlexItem>
                  <EuiFlexItem>
                    <EuiFlexGroup justifyContent="spaceBetween" gutterSize="s">
                      <EuiFlexItem grow={false}>
                        <EuiText size="s" color="subdued">
                          <strong>{dict.lastConnectedAt}:</strong>
                        </EuiText>
                      </EuiFlexItem>
                      <EuiFlexItem>
                        <EuiText size="s" textAlign="right">
                          {lastConnectedAt ? formatDate(lastConnectedAt, 'dobLong') : '-'}
                        </EuiText>
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPanel>
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiPanel>
      );
    }
    setItemIdToExpandedRowMap(itemIdToExpandedRowMapValues);
  };

  // Table columns
  const columns: Array<EuiBasicTableColumn<AP>> = [
    {
      field: 'name',
      name: dict.name,
      sortable: true,
      truncateText: true,
      mobileOptions: {
        render: (ap: AP) => (
          <>
            <strong>{ap.name}</strong>
            <br />
            <EuiHealth color={ap.online ? 'green' : 'danger'}>
              {ap.online ? dict.online : dict.offline}
            </EuiHealth>
          </>
        ),
        header: false,
        enlarge: true,
        width: '100%',
      },
    },
    {
      field: 'online',
      name: 'Status',
      sortable: true,
      render: (online: AP['online'], ap: AP) => (
        <EuiHealth color={online ? 'green' : 'danger'}>
          {online ? dict.online : dict.offline} <CheckQuality value={ap.signal || 0} />
        </EuiHealth>
      ),
      mobileOptions: { show: false },
    },
    {
      field: 'firstConnectedAt',
      name: dict.firstConnectedAt,
      dataType: 'date',
      sortable: true,
      render: (firstConnectedAt: AP['firstConnectedAt']) =>
        formatDate(firstConnectedAt, 'dobLong'),
      mobileOptions: { show: false },
    },
    {
      field: 'lastConnectedAt',
      name: dict.lastConnectedAt,
      dataType: 'date',
      sortable: true,
      render: (lastConnectedAt: AP['lastConnectedAt']) =>
        formatDate(lastConnectedAt, 'dobLong'),
      mobileOptions: { show: false },
    },
    {
      field: 'connectedDevices',
      name: dict.connectedDevices,
      align: 'center',
      sortable: true,
      render: (connectedDevices: AP['connectedDevices']) => connectedDevices || '0',
      mobileOptions: { show: false },
    },
    {
      align: 'right',
      width: '40px',
      isExpander: true,
      name: (
        <EuiScreenReaderOnly>
          <span>Expand row</span>
        </EuiScreenReaderOnly>
      ),
      mobileOptions: { header: false },
      render: (accessPoint: AP) => (
        <EuiButtonIcon
          onClick={() => toggleDetails(accessPoint)}
          aria-label={itemIdToExpandedRowMap[accessPoint.id] ? 'Collapse' : 'Expand'}
          iconType={itemIdToExpandedRowMap[accessPoint.id] ? 'arrowDown' : 'arrowRight'}
        />
      ),
    },
  ];

  // Table change handler
  const onTableChange = ({ page, sort }: Criteria<AP>) => {
    if (page) {
      const { index: pageIndex, size: pageSize } = page;
      setPageIndex(pageIndex);
      setPageSize(pageSize);
    }
    if (sort) {
      const { field: sortField, direction: sortDirection } = sort;
      setSortField(sortField);
      setSortDirection(sortDirection);
    }
  };

  // Pagination and sorting logic
  const findAps = (
    aps: AP[],
    pageIndex: number,
    pageSize: number,
    sortField: keyof AP,
    sortDirection: 'asc' | 'desc'
  ) => {
    let items = aps;

    if (sortField) {
      items = [...aps].sort(
        Comparators.property(sortField, Comparators.default(sortDirection))
      );
    }

    const startIndex = pageIndex * pageSize;
    const pageOfItems = items.slice(startIndex, Math.min(startIndex + pageSize, aps.length));

    return { pageOfItems, totalItemCount: aps.length };
  };

  const { pageOfItems, totalItemCount } = findAps(aps, pageIndex, pageSize, sortField, sortDirection);

  const sorting: EuiTableSortingType<AP> = {
    sort: {
      field: sortField,
      direction: sortDirection,
    },
  };

  const pagination = {
    pageIndex,
    pageSize,
    totalItemCount,
    pageSizeOptions: [5, 10, 20, 50],
  };

  // Loading state
  if (!domReady) {
    return (
      <EuiPage paddingSize="none">
        <EuiPageBody>
          <EuiLoadingSpinner size="xl" />
        </EuiPageBody>
      </EuiPage>
    );
  }

  return (
    <AuthCheck>
      <Navigation>
        <EuiPage paddingSize="l">
          <EuiPageBody>
            {/* Page Header */}
            <EuiPageHeader>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="s">
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="globe" size="xl" />
                  </EuiFlexItem>
                  <EuiFlexItem grow={false}>
                    <EuiTitle size="l">
                      <h1>{dict.accessPoint}</h1>
                    </EuiTitle>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="m">
                  <EuiFlexItem grow={false}>
                    <EuiButton
                      iconType="refresh"
                      onClick={fetchAPs}
                      aria-label="Refresh access points"
                    >
                      Refresh
                    </EuiButton>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
            </EuiPageHeader>

            <EuiSpacer size="l" />

            {/* Auto-refresh Toggle */}
            <EuiPanel>
              <EuiSwitch
                label={dict.refreshData}
                checked={auto}
                onChange={() => {
                  setAuto(!auto);
                  addToast({
                    title: auto ? dict.refreshDataDisabled : dict.refreshDataEnabled,
                    color: "primary",
                  });
                }}
              />
            </EuiPanel>

            <EuiSpacer size="l" />

            {/* Access Points Table */}
            <EuiPanel>
              {aps.length === 0 ? (
                <EuiEmptyPrompt
                  icon={<EuiIcon type="globe" size="xxl" />}
                  title={<h2>No Access Points Found</h2>}
                  body={
                    <p>
                      There are currently no access points configured. Access points will appear here once they connect to the network.
                    </p>
                  }
                />
              ) : (
                <EuiBasicTable
                  tableCaption="Access Points List"
                  items={pageOfItems}
                  itemId="id"
                  itemIdToExpandedRowMap={itemIdToExpandedRowMap}
                  columns={columns}
                  sorting={sorting}
                  pagination={pagination}
                  onChange={onTableChange}
                  responsive={true}
                />
              )}
            </EuiPanel>
          </EuiPageBody>
        </EuiPage>
      </Navigation>
    </AuthCheck>
  );
}
