import React from 'react';
import {
  EuiOverlayMask,
  EuiModal,
  EuiModalHeader,
  EuiModalHeaderTitle,
  EuiModalBody,
  EuiModalFooter,
  EuiFormRow,
  EuiFieldText,
  EuiSpacer,
  EuiText,
  EuiButton
} from '@elastic/eui';
import getDict from '@/lib/dict';
import { useFocusTrap } from '@/shared/hooks';
import { sanitizeInput } from '@/lib/security/validation';

const dict = getDict("de");

interface AccessPoint {
  id: string;
  [key: string]: any;
}

interface SaveProjectModalProps {
  show: boolean;
  onClose: () => void;
  projectName: string;
  setProjectName: (name: string) => void;
  onSave: () => void;
  apiLoading: boolean;
  accessPoints: AccessPoint[];
  pdfFile: File | null;
}

export const SaveProjectModal: React.FC<SaveProjectModalProps> = ({
  show,
  onClose,
  projectName,
  setProjectName,
  onSave,
  apiLoading,
  accessPoints,
  pdfFile
}) => {
  const modalRef = useFocusTrap(show);
  const [validationError, setValidationError] = React.useState<string>('');

  // Validate project name
  const validateProjectName = (name: string): boolean => {
    const trimmedName = name.trim();

    if (trimmedName.length === 0) {
      setValidationError('Project name is required');
      return false;
    }

    if (trimmedName.length < 2) {
      setValidationError('Project name must be at least 2 characters');
      return false;
    }

    if (trimmedName.length > 64) {
      setValidationError('Project name must not exceed 64 characters');
      return false;
    }

    if (!/^[a-zA-Z0-9\s\-_]+$/.test(trimmedName)) {
      setValidationError('Project name can only contain letters, numbers, spaces, hyphens, and underscores');
      return false;
    }

    setValidationError('');
    return true;
  };

  // Validate on name change
  React.useEffect(() => {
    if (projectName) {
      validateProjectName(projectName);
    }
  }, [projectName]);

  // Handle save with validation and sanitization
  const handleSave = () => {
    if (validateProjectName(projectName)) {
      onSave();
    }
  };

  const isFormValid = projectName.trim().length >= 2 && !validationError && !!pdfFile;

  if (!show) return null;

  return (
    <EuiOverlayMask>
      <EuiModal onClose={onClose} maxWidth={400} ref={modalRef as any}>
        <EuiModalHeader>
          <EuiModalHeaderTitle>{dict.saveProject}</EuiModalHeaderTitle>
        </EuiModalHeader>

        <EuiModalBody>
          <EuiFormRow
            label={dict.projectNameText}
            helpText={dict.name}
            error={validationError}
            isInvalid={!!validationError}
          >
            <EuiFieldText
              value={projectName}
              onChange={(e) => setProjectName(e.target.value)}
              placeholder={dict.name}
              onKeyDown={(e) => e.key === 'Enter' && handleSave()}
              isInvalid={!!validationError}
            />
          </EuiFormRow>

          <EuiSpacer size="m" />

          <EuiText size="s" color="subdued">
            <ul>
              <li>{accessPoints.length} {dict.accessPoint}</li>
              {pdfFile ? (
                <li>PDF file "{pdfFile.name}"</li>
              ) : (
                <li><strong>{dict.uploadText} {dict.accessPoint}</strong></li>
              )}
            </ul>
          </EuiText>
        </EuiModalBody>

        <EuiModalFooter>
          <EuiButton onClick={onClose}>
            {dict.cancelBtn}
          </EuiButton>
          <EuiButton
            onClick={handleSave}
            fill
            disabled={!isFormValid}
            isLoading={apiLoading}
          >
            {apiLoading ? dict.saving +'...' : dict.saveProject}
          </EuiButton>
        </EuiModalFooter>
      </EuiModal>
    </EuiOverlayMask>
  );
};
