/**
 * Stats Component
 * Professional statistics display for network metrics
 *
 * Improvements:
 * - Color-coded stat cards matching guest network design
 * - EUI icons instead of Lucide
 * - Better visual hierarchy with titled panel
 * - Consistent with modern EUI design patterns
 */

import React from 'react';
// External Libraries Dependencies
import {
  EuiStat,
  EuiFlexItem,
  EuiFlexGroup,
  EuiPanel,
  EuiSpacer,
  EuiTitle,
  EuiIcon,
} from '@elastic/eui';
// Own Dependencies
import Network from "@/app/types/Network";

export default function Stats(props: { network: Network }) {
  return (
    <EuiPanel>
      <EuiTitle size="s">
        <h3>Network Statistics</h3>
      </EuiTitle>
      <EuiSpacer size="m" />

      <EuiFlexGroup gutterSize="l" wrap>
        {/* Client Statistics */}
        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="accent" hasShadow={false}>
            <EuiStat
              title={props.network.total_clients}
              description="Total Clients"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="users" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="success" hasShadow={false}>
            <EuiStat
              title={props.network.online_clients}
              titleColor="success"
              description="Online"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="check" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="danger" hasShadow={false}>
            <EuiStat
              title={props.network.offline_clients}
              titleColor="danger"
              description="Offline"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="cross" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="primary" hasShadow={false}>
            <EuiStat
              title={props.network.authenticated_clients}
              description="Authenticated"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="lock" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="m" />

      {/* Traffic Statistics */}
      <EuiFlexGroup gutterSize="l" wrap>
        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="success" hasShadow={false}>
            <EuiStat
              title={(props.network.total_download / 1024).toFixed(2) + ' MB'}
              description="Total Download"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="sortDown" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="warning" hasShadow={false}>
            <EuiStat
              title={(props.network.total_upload / 1024).toFixed(2) + ' MB'}
              description="Total Upload"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="sortUp" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
          <EuiPanel color="primary" hasShadow={false}>
            <EuiStat
              title={((props.network.total_download + props.network.total_upload) / 1024).toFixed(2) + ' MB'}
              description="Total Traffic"
              titleSize="s"
              textAlign="center"
            >
              <EuiIcon type="storage" size="xl" />
            </EuiStat>
          </EuiPanel>
        </EuiFlexItem>
      </EuiFlexGroup>
    </EuiPanel>
  );
}
