import React, { useCallback, useRef, useState } from 'react';
import {
  EuiTourStep,
  EuiButton,
  EuiFlexGroup,
  EuiFlexItem,
  EuiText,
  EuiButtonIcon
} from '@elastic/eui';
import { ViewportControlsProps } from './types';
import { ZOOM_CONSTANTS, TOUR_STEPS } from './constants';
import getDict from '@/lib/dict';

const dict = getDict("de");
const { MIN_ZOOM, MAX_ZOOM } = ZOOM_CONSTANTS;

export const ViewportControls = React.memo(({
  state,
  dispatch,
  containerRef,
  onCropToViewport,
  isCropping
}: ViewportControlsProps) => {
  const [isTourActive, setIsTourActive] = useState(true);
  const [currentStep, setCurrentStep] = useState(0);
  const zoomControlsRef = useRef<HTMLDivElement>(null);
  const rotationControlsRef = useRef<HTMLDivElement>(null);
  const fitButtonRef = useRef<HTMLButtonElement>(null);
  const cropButtonRef = useRef<HTMLButtonElement>(null);

  const handleFitToView = useCallback(() => {
    if (containerRef.current) {
      const rect = containerRef.current.getBoundingClientRect();
      dispatch({
        type: 'FIT_TO_VIEW',
        payload: {
          containerWidth: rect.width,
          containerHeight: rect.height
        }
      });
    }
  }, [dispatch, containerRef]);

  const handleFinishTour = useCallback(() => {
    setIsTourActive(false);
    setCurrentStep(0);
  }, []);

  const handleNextStep = useCallback(() => {
    if (currentStep < TOUR_STEPS.length - 1) {
      setCurrentStep(currentStep + 1);
    } else {
      handleFinishTour();
    }
  }, [currentStep, handleFinishTour]);

  const handlePrevStep = useCallback(() => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  }, [currentStep]);

  const zoomPercentage = Math.round(state.viewport.zoomLevel * 100);
  const canZoomIn = state.viewport.zoomLevel < MAX_ZOOM;
  const canZoomOut = state.viewport.zoomLevel > MIN_ZOOM;
  const isCropDisabled = isCropping || (
    state.viewport.zoomLevel === 1 &&
    state.viewport.panOffset.x === 0 &&
    state.viewport.panOffset.y === 0
  );

  return (
    <div style={{
      position: 'absolute',
      top: '12px',
      right: '12px',
      display: 'flex',
      flexDirection: 'column',
      gap: '8px',
      zIndex: 1000
    }}>
      {/* Help Button */}
      <div style={{ display: 'flex', justifyContent: 'flex-end' }}>
        <EuiButtonIcon
          iconType="questionInCircle"
          color="primary"
          aria-label="Hilfe anzeigen"
          onClick={() => {
            setIsTourActive(true);
            setCurrentStep(0);
          }}
          display="fill"
          size="s"
        />
      </div>

      {/* Zoom Controls with Tour */}
      <EuiTourStep
        content={
          <div>
            <EuiText size="s">
              <p>{TOUR_STEPS[0].content}</p>
            </EuiText>
          </div>
        }
        isStepOpen={isTourActive && currentStep === 0}
        minWidth={300}
        onFinish={handleFinishTour}
        step={1}
        stepsTotal={TOUR_STEPS.length}
        title={TOUR_STEPS[0].title}
        anchorPosition="leftCenter"
        footerAction={
          <EuiFlexGroup justifyContent="spaceBetween">
            <EuiFlexItem grow={false}>
              <EuiButton size="s" color="text" onClick={handleFinishTour}>
                Überspringen
              </EuiButton>
            </EuiFlexItem>
            <EuiFlexItem grow={false}>
              <EuiButton size="s" fill onClick={handleNextStep}>
                Weiter
              </EuiButton>
            </EuiFlexItem>
          </EuiFlexGroup>
        }
      >
        <div
          ref={zoomControlsRef}
          style={{
            display: 'flex',
            backgroundColor: 'white',
            borderRadius: '6px',
            boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
            overflow: 'hidden',
            border: '1px solid #e1e5e9'
          }}
        >
          <button
            onClick={() => dispatch({ type: 'ZOOM_OUT' })}
            disabled={!canZoomOut}
            style={{
              padding: '8px 12px',
              border: 'none',
              backgroundColor: 'white',
              cursor: canZoomOut ? 'pointer' : 'not-allowed',
              opacity: canZoomOut ? 1 : 0.5,
              fontSize: '16px',
              fontWeight: 'bold'
            }}
            title="Zoom Out (Ctrl + -)"
          >
            −
          </button>

          <div style={{
            padding: '8px 16px',
            backgroundColor: '#f8f9fa',
            display: 'flex',
            alignItems: 'center',
            fontSize: '13px',
            fontWeight: '500',
            minWidth: '60px',
            justifyContent: 'center',
            borderLeft: '1px solid #e1e5e9',
            borderRight: '1px solid #e1e5e9'
          }}>
            {zoomPercentage}%
          </div>

          <button
            onClick={() => dispatch({ type: 'ZOOM_IN' })}
            disabled={!canZoomIn}
            style={{
              padding: '8px 12px',
              border: 'none',
              backgroundColor: 'white',
              cursor: canZoomIn ? 'pointer' : 'not-allowed',
              opacity: canZoomIn ? 1 : 0.5,
              fontSize: '16px',
              fontWeight: 'bold'
            }}
            title="Zoom In (Ctrl + +)"
          >
            +
          </button>
        </div>
      </EuiTourStep>

      {/* Rotation Controls with Tour */}
      <EuiTourStep
        content={
          <div>
            <EuiText size="s">
              <p>{TOUR_STEPS[1].content}</p>
            </EuiText>
          </div>
        }
        isStepOpen={isTourActive && currentStep === 1}
        minWidth={300}
        onFinish={handleFinishTour}
        step={2}
        stepsTotal={TOUR_STEPS.length}
        title={TOUR_STEPS[1].title}
        anchorPosition="leftCenter"
        footerAction={
          <EuiFlexGroup justifyContent="spaceBetween">
            <EuiFlexItem grow={false}>
              <EuiButton size="s" color="text" onClick={handlePrevStep}>
                Zurück
              </EuiButton>
            </EuiFlexItem>
            <EuiFlexItem grow={false}>
              <EuiButton size="s" fill onClick={handleNextStep}>
                Weiter
              </EuiButton>
            </EuiFlexItem>
          </EuiFlexGroup>
        }
      >
        <div
          ref={rotationControlsRef}
          style={{
            display: 'flex',
            backgroundColor: 'white',
            borderRadius: '6px',
            boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
            overflow: 'hidden',
            border: '1px solid #e1e5e9'
          }}
        >
          <button
            onClick={() => dispatch({ type: 'ROTATE_LEFT' })}
            style={{
              padding: '8px 12px',
              border: 'none',
              backgroundColor: 'white',
              cursor: 'pointer',
              fontSize: '16px'
            }}
            title="Rotate Left (Ctrl + ←)"
          >
            ↻
          </button>

          <div style={{
            padding: '8px 12px',
            backgroundColor: '#f8f9fa',
            display: 'flex',
            alignItems: 'center',
            fontSize: '12px',
            fontWeight: '500',
            minWidth: '45px',
            justifyContent: 'center',
            borderLeft: '1px solid #e1e5e9',
            borderRight: '1px solid #e1e5e9'
          }}>
            {state.viewport.rotation}°
          </div>

          <button
            onClick={() => dispatch({ type: 'ROTATE_RIGHT' })}
            style={{
              padding: '8px 12px',
              border: 'none',
              backgroundColor: 'white',
              cursor: 'pointer',
              fontSize: '16px'
            }}
            title="Rotate Right (Ctrl + →)"
          >
            ↺
          </button>
        </div>
      </EuiTourStep>

      {/* Action Controls */}
      <div style={{
        display: 'flex',
        flexDirection: 'column',
        gap: '4px'
      }}>
        {/* Fit to View Button with Tour */}
        <EuiTourStep
          content={
            <div>
              <EuiText size="s">
                <p>{TOUR_STEPS[2].content}</p>
              </EuiText>
            </div>
          }
          isStepOpen={isTourActive && currentStep === 2}
          minWidth={300}
          onFinish={handleFinishTour}
          step={3}
          stepsTotal={TOUR_STEPS.length}
          title={TOUR_STEPS[2].title}
          anchorPosition="leftCenter"
          footerAction={
            <EuiFlexGroup justifyContent="spaceBetween">
              <EuiFlexItem grow={false}>
                <EuiButton size="s" color="text" onClick={handlePrevStep}>
                  Zurück
                </EuiButton>
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButton size="s" fill onClick={handleNextStep}>
                  Weiter
                </EuiButton>
              </EuiFlexItem>
            </EuiFlexGroup>
          }
        >
          <button
            ref={fitButtonRef}
            onClick={handleFitToView}
            style={{
              padding: '8px 12px',
              border: '1px solid #e1e5e9',
              backgroundColor: 'white',
              borderRadius: '6px',
              cursor: 'pointer',
              fontSize: '12px',
              fontWeight: '500',
              boxShadow: '0 2px 8px rgba(0,0,0,0.15)'
            }}
            title="Fit to View (Ctrl + 0)"
          >
            {dict.fitToView}
          </button>
        </EuiTourStep>

        {/* Crop Button with Tour */}
        <EuiTourStep
          content={
            <div>
              <EuiText size="s">
                <p>{TOUR_STEPS[3].content}</p>
              </EuiText>
            </div>
          }
          isStepOpen={isTourActive && currentStep === 3}
          minWidth={300}
          onFinish={handleFinishTour}
          step={4}
          stepsTotal={TOUR_STEPS.length}
          title={TOUR_STEPS[3].title}
          anchorPosition="leftCenter"
          footerAction={
            <EuiFlexGroup justifyContent="spaceBetween">
              <EuiFlexItem grow={false}>
                <EuiButton size="s" color="text" onClick={handlePrevStep}>
                  Zurück
                </EuiButton>
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiButton size="s" fill onClick={handleFinishTour}>
                  Fertig
                </EuiButton>
              </EuiFlexItem>
            </EuiFlexGroup>
          }
        >
          <button
            ref={cropButtonRef}
            onClick={onCropToViewport}
            disabled={isCropDisabled}
            style={{
              padding: '8px 12px',
              border: '1px solid #e1e5e9',
              backgroundColor: 'white',
              borderRadius: '6px',
              cursor: isCropDisabled ? 'not-allowed' : 'pointer',
              fontSize: '12px',
              fontWeight: '500',
              boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
              opacity: isCropDisabled ? 0.5 : 1
            }}
            title="Crop"
          >
            {dict.crop}
          </button>
        </EuiTourStep>
      </div>
    </div>
  );
});

ViewportControls.displayName = 'ViewportControls';
