"use client"

import React, { useReducer, useCallback, useRef } from 'react';
import { EuiButton } from '@elastic/eui';
import { SCREEN_TYPES } from '@/features/project-setup/utils/constants';
import getDict from '@/lib/dict';
import { ViewportControls } from './ViewportControls';
import { viewportReducer, initialState } from './reducer';
import { useMouseControls } from './hooks/useMouseControls';
import { useTouchControls } from './hooks/useTouchControls';
import { useCropHandler } from './hooks/useCropHandler';
import { ZoomPanRotationEditorProps } from './types';

const dict = getDict("de");

export default function ZoomPanRotationEditor({
  imgsrc,
  navigateTo,
  dd,
  setPdfFile
}: ZoomPanRotationEditorProps) {
  const [state, dispatch] = useReducer(viewportReducer, initialState);
  const imageRef = useRef<HTMLImageElement>(null);
  const containerRef = useRef<HTMLDivElement>(null);

  // Handle image load
  const handleImageLoad = useCallback(() => {
    if (imageRef.current) {
      const { offsetWidth, offsetHeight } = imageRef.current;
      dispatch({
        type: 'SET_IMAGE_DIMENSIONS',
        payload: { width: offsetWidth, height: offsetHeight }
      });
      if (containerRef.current) {
        const rect = containerRef.current.getBoundingClientRect();
        dispatch({
          type: 'FIT_TO_VIEW',
          payload: {
            containerWidth: rect.width,
            containerHeight: rect.height
          }
        });
      }
    }
  }, []);

  // Use custom hooks for interaction controls
  const { handleWheel, handleMouseDown } = useMouseControls({
    state,
    dispatch,
    containerRef
  });

  const { handleTouchStart, handleTouchMove, handleTouchEnd } = useTouchControls({
    state,
    dispatch
  });

  const { handleCropToViewport, isCropping } = useCropHandler({
    state,
    dispatch,
    imageRef,
    containerRef,
    dd,
    setPdfFile
  });

  return (
    <div style={{
      fontFamily: 'Inter, -apple-system, BlinkMacSystemFont, sans-serif',
      height: '100vh',
      display: 'flex',
      flexDirection: 'column',
      backgroundColor: '#f5f7fa'
    }}>
      {/* Header */}
      <div style={{
        padding: '16px',
        backgroundColor: 'white',
        borderBottom: '1px solid #e1e5e9',
        display: 'flex',
        justifyContent: 'space-between',
        alignItems: 'center'
      }}>
        <div>
          <h1 style={{ margin: '0 0 4px 0', fontSize: '20px', fontWeight: '600' }}>
            {dict.cropText}
          </h1>
        </div>
        <EuiButton
          color={'success'}
          onClick={() => {
            navigateTo(SCREEN_TYPES.PROJECT_SETUP);
          }}
        >
          {dict.continue}
        </EuiButton>
      </div>

      {/* Main container */}
      <div
        ref={containerRef}
        style={{
          flex: 1,
          position: 'relative',
          overflow: 'hidden',
          cursor: state.viewport.isPanning ? 'grabbing' : 'grab'
        }}
        onWheel={(e) => handleWheel(e.nativeEvent)}
        onMouseDown={handleMouseDown}
        onTouchStart={handleTouchStart}
        onTouchMove={handleTouchMove}
        onTouchEnd={handleTouchEnd}
      >
        {/* Transformed content */}
        <div
          style={{
            position: 'absolute',
            top: 0,
            left: 0,
            transform: `translate(${state.viewport.panOffset.x}px, ${state.viewport.panOffset.y}px) scale(${state.viewport.zoomLevel})`,
            transformOrigin: '0 0',
            transition: state.viewport.isPanning ? 'none' : 'transform 0.1s ease-out'
          }}
        >
          <div style={{
            transform: `rotate(${state.viewport.rotation}deg)`,
            transformOrigin: 'center center'
          }}>
            <img
              ref={imageRef}
              src={imgsrc}
              alt="Floor Plan"
              style={{
                display: 'block',
                userSelect: 'none',
                maxWidth: 'none'
              }}
              onLoad={handleImageLoad}
              draggable={false}
            />
          </div>
        </div>

        {/* Viewport Controls */}
        <ViewportControls
          state={state}
          dispatch={dispatch}
          containerRef={containerRef}
          onCropToViewport={handleCropToViewport}
          isCropping={isCropping}
        />
      </div>
    </div>
  );
}
