import { useState, useCallback } from 'react';
import { cropImageToViewport } from '@/features/pdf-editor/utils/pdfUtils';
import { ViewportAction, ZoomPanState } from '../types';

interface UseCropHandlerProps {
  state: ZoomPanState;
  dispatch: React.Dispatch<ViewportAction>;
  imageRef: React.RefObject<HTMLImageElement>;
  containerRef: React.RefObject<HTMLDivElement>;
  dd: (action: { type: string; payload?: unknown }) => void;
  setPdfFile: (file: File | null) => void;
}

export function useCropHandler({
  state,
  dispatch,
  imageRef,
  containerRef,
  dd,
  setPdfFile
}: UseCropHandlerProps) {
  const [originalImageSrc, setOriginalImageSrc] = useState<string | null>(null);
  const [isCropping, setIsCropping] = useState(false);

  const handleCropToViewport = useCallback(async () => {
    if (isCropping) return;

    setIsCropping(true);

    try {
      // Store original image source if not already stored
      if (!originalImageSrc && imageRef.current) {
        setOriginalImageSrc(imageRef.current.src);
      }

      const cropResult = await cropImageToViewport(
        imageRef,
        state.viewport,
        state.imageDimensions,
        containerRef,
        dd,
        setPdfFile
      );

      if (cropResult) {
        // Update the image source
        if (imageRef.current) {
          imageRef.current.src = cropResult.dataUrl;
        }
        dd({
          type: 'UPDATE_SRC',
          payload: cropResult.dataUrl
        });

        // Update state
        dispatch({
          type: 'UPDATE_AFTER_CROP',
          payload: {
            imageDimensions: cropResult.newDimensions
          }
        });

        // Reset viewport
        dispatch({ type: 'RESET_VIEW' });
      }
    } catch (error) {
      console.error('Error cropping image:', error);
    } finally {
      setIsCropping(false);
    }
  }, [
    state.viewport,
    state.imageDimensions,
    originalImageSrc,
    isCropping,
    imageRef,
    containerRef,
    dd,
    setPdfFile,
    dispatch
  ]);

  return {
    handleCropToViewport,
    isCropping,
    originalImageSrc
  };
}
