import { useCallback, useEffect } from 'react';
import { ViewportAction, ZoomPanState } from '../types';
import { ZOOM_CONSTANTS } from '../constants';

const { MIN_ZOOM, MAX_ZOOM, ZOOM_SENSITIVITY } = ZOOM_CONSTANTS;

interface UseMouseControlsProps {
  state: ZoomPanState;
  dispatch: React.Dispatch<ViewportAction>;
  containerRef: React.RefObject<HTMLDivElement>;
}

export function useMouseControls({ state, dispatch, containerRef }: UseMouseControlsProps) {
  // Mouse wheel zoom
  const handleWheel = useCallback((e: WheelEvent) => {
    if (e.ctrlKey || e.metaKey) {
      e.preventDefault();

      if (!containerRef.current) return;

      const rect = containerRef.current.getBoundingClientRect();
      const mouseX = e.clientX - rect.left;
      const mouseY = e.clientY - rect.top;

      // Calculate zoom factor
      const zoomFactor = -e.deltaY * ZOOM_SENSITIVITY;
      const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, state.viewport.zoomLevel + zoomFactor));

      if (newZoom !== state.viewport.zoomLevel) {
        // Zoom towards mouse position
        const zoomRatio = newZoom / state.viewport.zoomLevel;
        const newPanX = mouseX - (mouseX - state.viewport.panOffset.x) * zoomRatio;
        const newPanY = mouseY - (mouseY - state.viewport.panOffset.y) * zoomRatio;

        dispatch({ type: 'SET_ZOOM', payload: newZoom });
        dispatch({ type: 'SET_PAN', payload: { x: newPanX, y: newPanY } });
      }
    }
  }, [state.viewport.zoomLevel, state.viewport.panOffset, dispatch, containerRef]);

  // Pan controls
  const handleMouseDown = useCallback((e: React.MouseEvent) => {
    if (e.button === 1 || e.button === 0) {
      e.preventDefault();
      dispatch({
        type: 'START_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
    }
  }, [dispatch]);

  const handleMouseMove = useCallback((e: MouseEvent) => {
    if (state.viewport.isPanning) {
      dispatch({
        type: 'UPDATE_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
    }
  }, [state.viewport.isPanning, dispatch]);

  const handleMouseUp = useCallback(() => {
    if (state.viewport.isPanning) {
      dispatch({ type: 'END_PAN' });
    }
  }, [state.viewport.isPanning, dispatch]);

  // Event listeners
  useEffect(() => {
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);

    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [handleMouseMove, handleMouseUp]);

  return {
    handleWheel,
    handleMouseDown,
  };
}
