import { useCallback } from 'react';
import { ViewportAction, ZoomPanState } from '../types';
import { getTouchDistance } from '../utils';
import { ZOOM_CONSTANTS } from '../constants';

const { MIN_ZOOM, MAX_ZOOM } = ZOOM_CONSTANTS;

interface UseTouchControlsProps {
  state: ZoomPanState;
  dispatch: React.Dispatch<ViewportAction>;
}

export function useTouchControls({ state, dispatch }: UseTouchControlsProps) {
  const handleTouchStart = useCallback((e: React.TouchEvent) => {
    if (e.touches.length === 2) {
      e.preventDefault();
      const distance = getTouchDistance(e.touches[0], e.touches[1]);
      dispatch({
        type: 'SET_TOUCH',
        payload: {
          initialDistance: distance,
          initialZoom: state.viewport.zoomLevel,
          isPinching: true,
          touchCount: 2
        }
      });
    } else if (e.touches.length === 1) {
      const touch = e.touches[0];
      dispatch({
        type: 'START_PAN',
        payload: { x: touch.clientX, y: touch.clientY }
      });
      dispatch({
        type: 'SET_TOUCH',
        payload: { touchCount: 1 }
      });
    }
  }, [state.viewport.zoomLevel, dispatch]);

  const handleTouchMove = useCallback((e: React.TouchEvent) => {
    if (e.touches.length === 2 && state.touch.isPinching) {
      e.preventDefault();
      const distance = getTouchDistance(e.touches[0], e.touches[1]);
      const scale = distance / (state.touch.initialDistance || 1);
      const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, state.touch.initialZoom * scale));

      dispatch({ type: 'SET_ZOOM', payload: newZoom });
    } else if (e.touches.length === 1 && state.viewport.isPanning) {
      e.preventDefault();
      const touch = e.touches[0];
      dispatch({
        type: 'UPDATE_PAN',
        payload: { x: touch.clientX, y: touch.clientY }
      });
    }
  }, [state.touch, state.viewport.isPanning, dispatch]);

  const handleTouchEnd = useCallback((e: React.TouchEvent) => {
    if (e.touches.length < 2) {
      dispatch({
        type: 'SET_TOUCH',
        payload: { isPinching: false, initialDistance: null }
      });
    }
    if (e.touches.length === 0) {
      dispatch({ type: 'END_PAN' });
      dispatch({
        type: 'SET_TOUCH',
        payload: { touchCount: 0 }
      });
    }
  }, [dispatch]);

  return {
    handleTouchStart,
    handleTouchMove,
    handleTouchEnd,
  };
}
