import { ZoomPanState, ViewportAction } from './types';
import { ZOOM_CONSTANTS, ROTATION_CONSTANTS, VIEWPORT_CONSTANTS } from './constants';

const { MIN_ZOOM, MAX_ZOOM, ZOOM_STEP } = ZOOM_CONSTANTS;
const { ROTATE_STEP, FULL_ROTATION } = ROTATION_CONSTANTS;
const { CONTAINER_PADDING } = VIEWPORT_CONSTANTS;

export const initialState: ZoomPanState = {
  viewport: {
    zoomLevel: 1,
    panOffset: { x: 0, y: 0 },
    rotation: 0,
    isPanning: false,
    lastPanPosition: { x: 0, y: 0 }
  },
  pdfFile: null,
  accessPoints: [],
  imageDimensions: { width: 0, height: 0 },
  draggedItem: null,
  touch: {
    initialDistance: null,
    initialZoom: 1,
    isPinching: false,
    touchCount: 0
  },
  ui: {
    showControls: false,
  }
};

export function viewportReducer(state: ZoomPanState, action: ViewportAction): ZoomPanState {
  switch (action.type) {
  case 'SET_ZOOM':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, action.payload))
      }
    };

  case 'ZOOM_IN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.min(MAX_ZOOM, state.viewport.zoomLevel + ZOOM_STEP)
      }
    };

  case 'ZOOM_OUT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.max(MIN_ZOOM, state.viewport.zoomLevel - ZOOM_STEP)
      }
    };

  case 'SET_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        panOffset: action.payload
      }
    };

  case 'START_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        isPanning: true,
        lastPanPosition: action.payload
      }
    };

  case 'UPDATE_PAN':
    if (!state.viewport.isPanning) return state;

    const deltaX = action.payload.x - state.viewport.lastPanPosition.x;
    const deltaY = action.payload.y - state.viewport.lastPanPosition.y;

    return {
      ...state,
      viewport: {
        ...state.viewport,
        panOffset: {
          x: state.viewport.panOffset.x + deltaX,
          y: state.viewport.panOffset.y + deltaY
        },
        lastPanPosition: action.payload
      }
    };

  case 'END_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        isPanning: false
      }
    };

  case 'SET_ROTATION':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: ((action.payload % FULL_ROTATION) + FULL_ROTATION) % FULL_ROTATION
      }
    };

  case 'ROTATE_LEFT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: ((state.viewport.rotation - ROTATE_STEP) + FULL_ROTATION) % FULL_ROTATION
      }
    };

  case 'ROTATE_RIGHT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: (state.viewport.rotation + ROTATE_STEP) % FULL_ROTATION
      }
    };

  case 'RESET_VIEW':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: 1,
        panOffset: { x: 0, y: 0 },
        rotation: 0
      }
    };

  case 'FIT_TO_VIEW':
    const containerWidth = action.payload.containerWidth - CONTAINER_PADDING;
    const containerHeight = action.payload.containerHeight - CONTAINER_PADDING;
    const imageWidth = state.imageDimensions.width;
    const imageHeight = state.imageDimensions.height;

    if (imageWidth === 0 || imageHeight === 0) return state;

    const scaleX = containerWidth / imageWidth;
    const scaleY = containerHeight / imageHeight;
    const fitZoom = Math.min(scaleX, scaleY, MAX_ZOOM);

    const panX = (containerWidth - imageWidth * fitZoom) / 2;
    const panY = (containerHeight - imageHeight * fitZoom) / 2;

    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: fitZoom,
        panOffset: { x: panX, y: panY },
        rotation: 0
      }
    };

  case 'SET_DRAGGED_ITEM':
    return {
      ...state,
      draggedItem: action.payload
    };

  case 'SET_PDF_FILE':
    return {
      ...state,
      pdfFile: action.payload
    };

  case 'SET_IMAGE_DIMENSIONS':
    return {
      ...state,
      imageDimensions: action.payload
    };

  case 'SET_TOUCH':
    return {
      ...state,
      touch: { ...state.touch, ...action.payload }
    };

  case 'CROP_TO_VIEWPORT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: 1,
        panOffset: { x: 0, y: 0 },
      },
      imageDimensions: action.payload.newDimensions
    };

  case 'UPDATE_AFTER_CROP':
    return {
      ...state,
      imageDimensions: action.payload.imageDimensions
    };

  default:
    return state;
  }
}
