import { ViewportState, ImageDimensions } from './types';

/**
 * Converts screen coordinates to image-relative coordinates
 * accounting for zoom, pan, and rotation transforms
 */
export function getImageRelativePosition(
  clientX: number,
  clientY: number,
  imageRef: React.RefObject<HTMLImageElement>,
  viewport: ViewportState,
  imageDimensions: ImageDimensions
): { x: number; y: number } {
  if (!imageRef.current) return { x: 0, y: 0 };

  const imageRect = imageRef.current.getBoundingClientRect();

  // Step 1: Convert screen coordinates to container coordinates
  const containerX = clientX - imageRect.left;
  const containerY = clientY - imageRect.top;

  // Step 2: Account for pan offset
  const panAdjustedX = (containerX - viewport.panOffset.x) / viewport.zoomLevel;
  const panAdjustedY = (containerY - viewport.panOffset.y) / viewport.zoomLevel;

  // Step 3: Account for rotation (inverse transform)
  if (viewport.rotation !== 0) {
    // Get image center
    const centerX = imageDimensions.width / 2;
    const centerY = imageDimensions.height / 2;

    // Translate to origin (center of image)
    const translatedX = panAdjustedX - centerX;
    const translatedY = panAdjustedY - centerY;

    // Apply inverse rotation
    const radians = (-viewport.rotation * Math.PI) / 180; // Negative for inverse
    const cos = Math.cos(radians);
    const sin = Math.sin(radians);

    const rotatedX = translatedX * cos - translatedY * sin;
    const rotatedY = translatedX * sin + translatedY * cos;

    // Translate back
    return {
      x: rotatedX + centerX,
      y: rotatedY + centerY
    };
  }

  return { x: panAdjustedX, y: panAdjustedY };
}

/**
 * Calculates distance between two touch points
 */
export function getTouchDistance(touch1: Touch, touch2: Touch): number {
  const dx = touch1.clientX - touch2.clientX;
  const dy = touch1.clientY - touch2.clientY;
  return Math.sqrt(dx * dx + dy * dy);
}
