"use client";

import Image from "next/image";
import Link from "next/link";
import branding from "@/lib/branding";
import { getFirstPathElement, MenuItem } from "./Navigation";
import getDict from "@/lib/dict";
import React from "react";
import { usePathname } from "next/navigation";
import Cookies from "js-cookie";
import { useAuthContext } from "@/features/auth";
import Icon from "@/app/components/Icon";
import {
  EuiHeader,
  EuiHeaderSection,
  EuiHeaderSectionItem,
  EuiButtonEmpty,
  EuiBadge,
  EuiText,
  EuiFlexGroup,
  EuiFlexItem,
  EuiIcon,
  EuiCollapsibleNavGroup,
  EuiFlyout,
  EuiFlyoutBody,
  EuiFlyoutHeader,
  EuiTitle,
  useEuiTheme,
  EuiSpacer,
} from "@elastic/eui";

export default function Navbar(props: { menuItems: MenuItem[] }) {
  const dict = getDict("de");
  const pathName = usePathname();
  const { logout, isLoggingOut, userEmail: authUserEmail } = useAuthContext();
  const { euiTheme } = useEuiTheme();

  const [navFlyoutOpen, setNavFlyoutOpen] = React.useState(false);

  const handleLogout = async () => {
    await logout();
  };
  const userEmail = authUserEmail || Cookies.get('username');

  // Auto-close flyout on resize
  React.useEffect(() => {
    const handleResize = () => {
      if (window.innerWidth > 800) {
        setNavFlyoutOpen(false);
      }
    };

    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  const closeFlyout = () => setNavFlyoutOpen(false);

  return (
    <>
      <EuiHeader
        position="fixed"
        style={{
          background: 'white',
          borderBottom: `1px solid ${euiTheme.colors.lightShade}`,
          boxShadow: '0 2px 4px rgba(0,0,0,0.05)',
        }}
      >
        {/* Left section - Logo */}
        <EuiHeaderSection grow={false}>
          <EuiHeaderSectionItem>
            <Link href="/" style={{ textDecoration: 'none', display: 'flex', alignItems: 'center', gap: '12px' }}>
              <Image
                src="/logo.svg"
                alt="Logo"
                width={32}
                height={32}
                priority
                style={{ display: 'block' }}
              />
              <EuiText size="m" style={{ fontWeight: 700, color: euiTheme.colors.text }}>
                <h1 style={{ margin: 0, fontSize: '1.25rem' }}>{branding.productName}</h1>
              </EuiText>
              <EuiBadge color="hollow">v2.0.0</EuiBadge>
            </Link>
          </EuiHeaderSectionItem>
        </EuiHeaderSection>

        {/* Right section - User info and menu */}
        <EuiHeaderSection side="right">
          {/* Desktop user info */}
          <EuiHeaderSectionItem
            className="desktop-user-info"
          >
            <EuiFlexGroup alignItems="center" gutterSize="m" responsive={false}>
              <EuiFlexItem grow={false}>
                <EuiFlexGroup alignItems="center" gutterSize="xs" responsive={false}>
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="user" size="m" color="subdued" />
                  </EuiFlexItem>
                  <EuiFlexItem grow={false}>
                    <EuiText size="s" color="subdued">
                      <span style={{ fontSize: '0.875rem' }}>
                        {dict.loggedInAs}: <strong style={{ color: euiTheme.colors.text }}>{userEmail}</strong>
                      </span>
                    </EuiText>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiFlexItem>

              <EuiFlexItem grow={false}>
                <EuiButtonEmpty
                  iconType="exit"
                  color="text"
                  onClick={handleLogout}
                  isDisabled={isLoggingOut}
                  isLoading={isLoggingOut}
                  size="s"
                  style={{ minWidth: '100px' }}
                >
                  {isLoggingOut ? 'Logging out...' : dict.logout}
                </EuiButtonEmpty>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiHeaderSectionItem>

          {/* Mobile menu button */}
          <EuiHeaderSectionItem className="mobile-menu-button">
            <EuiButtonEmpty
              iconType="menu"
              onClick={() => setNavFlyoutOpen(!navFlyoutOpen)}
              aria-label="Open navigation menu"
              color="text"
              size="s"
            >
              Menu
            </EuiButtonEmpty>
          </EuiHeaderSectionItem>
        </EuiHeaderSection>
      </EuiHeader>
      {/* Mobile navigation flyout */}
      {navFlyoutOpen && (
        <EuiFlyout
          onClose={closeFlyout}
          size="s"
          side="left"
          ownFocus
          hideCloseButton={false}
          aria-labelledby="flyoutTitle"
        >
          <EuiFlyoutHeader hasBorder>
            <EuiTitle size="s">
              <h2 id="flyoutTitle">Navigation</h2>
            </EuiTitle>
          </EuiFlyoutHeader>

          <EuiFlyoutBody>
            {/* User info section */}
            <EuiCollapsibleNavGroup
              title={
                <EuiFlexGroup alignItems="center" gutterSize="xs" responsive={false}>
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="user" size="m" />
                  </EuiFlexItem>
                  <EuiFlexItem>
                    <EuiText size="s">
                      <strong>{userEmail}</strong>
                    </EuiText>
                  </EuiFlexItem>
                </EuiFlexGroup>
              }
              isCollapsible={false}
              background="light"
            >
              <EuiButtonEmpty
                iconType="exit"
                color="danger"
                onClick={handleLogout}
                isDisabled={isLoggingOut}
                isLoading={isLoggingOut}
                flush="left"
                style={{ width: '100%' }}
              >
                {isLoggingOut ? 'Logging out...' : dict.logout}
              </EuiButtonEmpty>
            </EuiCollapsibleNavGroup>

            <EuiSpacer size="m" />

            {/* Navigation links */}
            <EuiCollapsibleNavGroup
              title="Menu"
              iconType="menu"
              isCollapsible={false}
            >
              {props.menuItems.map((item) => {
                const isActive = getFirstPathElement(pathName) === getFirstPathElement(item.href);

                return (
                  <Link
                    key={item.href}
                    href={item.href}
                    onClick={closeFlyout}
                    style={{ textDecoration: 'none', display: 'block' }}
                  >
                    <EuiFlexGroup
                      alignItems="center"
                      gutterSize="m"
                      responsive={false}
                      style={{
                        padding: '12px 16px',
                        borderRadius: euiTheme.border.radius.medium,
                        background: isActive ? euiTheme.colors.lightestShade : 'transparent',
                        borderLeft: isActive ? `3px solid ${euiTheme.colors.primary}` : '3px solid transparent',
                        paddingLeft: '13px',
                        cursor: 'pointer',
                        transition: 'all 0.2s ease',
                        marginBottom: '4px',
                      }}
                    >
                      <EuiFlexItem grow={false}>
                        <Icon
                          name={item.icon.name}
                          size="1.25rem"
                          color={isActive ? euiTheme.colors.primary : euiTheme.colors.subduedText}
                        />
                      </EuiFlexItem>
                      <EuiFlexItem>
                        <EuiText
                          size="m"
                          style={{
                            color: isActive ? euiTheme.colors.primary : euiTheme.colors.text,
                            fontWeight: isActive ? 600 : 400,
                          }}
                        >
                          {item.text}
                        </EuiText>
                      </EuiFlexItem>
                    </EuiFlexGroup>
                  </Link>
                );
              })}
            </EuiCollapsibleNavGroup>
          </EuiFlyoutBody>
        </EuiFlyout>
      )}

      <style jsx global>{`
        .desktop-user-info {
          display: none;
        }

        .mobile-menu-button {
          display: block;
        }

        @media (min-width: 801px) {
          .desktop-user-info {
            display: flex;
          }

          .mobile-menu-button {
            display: none;
          }
        }
      `}</style>
    </>
  );
}
