import React, { useCallback } from 'react';
import { MapCanvasProps, ViewportControlsProps } from './types';
import { AccessPointLayer } from '@/features/project-setup/components/ui/AccessPointLayer';
import { MAX_ZOOM, MIN_ZOOM } from '@/lib/nodeScaling';
import { Z_INDEX, SPACING, POSITION } from '@/lib/uiConstants';
import {
  CONTROL_PANEL_STYLES,
  BUTTON_STYLES,
  DISPLAY_STYLES,
  TOUCH_DRAG_STYLES,
  MAP_CONTAINER_STYLES,
  COLORS,
  getCursorStyle,
  getButtonCursor,
  getButtonOpacity,
  getTransition,
} from './mapCanvasConstants';


const ViewportControls = React.memo<ViewportControlsProps>(({ state, dispatch, containerRef }) => {

  const zoomPercentage = Math.round(state.viewport.zoomLevel * 100);
  const canZoomIn = state.viewport.zoomLevel < MAX_ZOOM;
  const canZoomOut = state.viewport.zoomLevel > MIN_ZOOM;

  return (
    <div style={{
      ...CONTROL_PANEL_STYLES.CONTAINER,
      top: `${POSITION.CONTROL_TOP}px`,
      right: `${POSITION.CONTROL_RIGHT}px`,
      gap: `${SPACING.GAP_SMALL}px`,
      zIndex: Z_INDEX.CONTROLS
    }}>
      {/* Zoom Controls */}
      <div style={CONTROL_PANEL_STYLES.BASE_CONTROL}>
        <button
          onClick={() => dispatch({ type: 'ZOOM_OUT' })}
          disabled={!canZoomOut}
          aria-label="Zoom out (Ctrl + Minus)"
          aria-disabled={!canZoomOut}
          title="Zoom Out (Ctrl + -)"
          style={{
            ...BUTTON_STYLES.BASE,
            ...BUTTON_STYLES.ZOOM,
            cursor: getButtonCursor(canZoomOut),
            opacity: getButtonOpacity(canZoomOut),
          }}
        >
          −
        </button>

        <div style={DISPLAY_STYLES.ZOOM_PERCENTAGE}>
          {zoomPercentage}%
        </div>

        <button
          onClick={() => dispatch({ type: 'ZOOM_IN' })}
          disabled={!canZoomIn}
          aria-label="Zoom in (Ctrl + Plus or Equals)"
          aria-disabled={!canZoomIn}
          title="Zoom In (Ctrl + +)"
          style={{
            ...BUTTON_STYLES.BASE,
            ...BUTTON_STYLES.ZOOM,
            cursor: getButtonCursor(canZoomIn),
            opacity: getButtonOpacity(canZoomIn),
          }}
        >
          +
        </button>
      </div>

      {/* Rotation Controls */}
      <div style={CONTROL_PANEL_STYLES.BASE_CONTROL}>
        <button
          onClick={() => dispatch({ type: 'ROTATE_LEFT' })}
          aria-label="Rotate left 90 degrees (Ctrl + Left Arrow)"
          title="Rotate Left (Ctrl + ←)"
          style={{
            ...BUTTON_STYLES.BASE,
            ...BUTTON_STYLES.ROTATION,
            cursor: 'pointer',
          }}
        >
          ↻
        </button>

        <div style={DISPLAY_STYLES.ROTATION_DEGREES}>
          {state.viewport.rotation}°
        </div>

        <button
          onClick={() => dispatch({ type: 'ROTATE_RIGHT' })}
          aria-label="Rotate right 90 degrees (Ctrl + Right Arrow)"
          title="Rotate Right (Ctrl + →)"
          style={{
            ...BUTTON_STYLES.BASE,
            ...BUTTON_STYLES.ROTATION,
            cursor: 'pointer',
          }}
        >
          ↺
        </button>
      </div>
    </div>
  );
});

ViewportControls.displayName = 'ViewportControls';

export const MapCanvas: React.FC<MapCanvasProps> = ({
  pdfPages,
  currentPage,
  accessPoints,
  accessPointTypes,
  viewport,
  imageDimensions,
  allowNodeMovement,
  editingNodeId,
  draggedItem,
  touchDragPreview,
  touchDragItem,
  src,
  onImageLoad,
  onDrop,
  onDragOver,
  onWheel,
  onMouseDown,
  onAccessPointDragStart,
  onTouchStartAccessPoint,
  startEditingNode,
  showNodeDetails,
  dispatch,
  containerRef,
  mapContainerRef,
  mapImageRef
}) => {
  const currentPageAccessPoints = accessPoints.filter(ap => ap.page === currentPage);

  // Keyboard navigation handler
  const handleKeyDown = useCallback((e: React.KeyboardEvent) => {
    if (e.ctrlKey || e.metaKey) {
      switch (e.key) {
      case '=':
      case '+':
        e.preventDefault();
        if (viewport.zoomLevel < MAX_ZOOM) {
          dispatch({ type: 'ZOOM_IN' });
        }
        break;
      case '-':
        e.preventDefault();
        if (viewport.zoomLevel > MIN_ZOOM) {
          dispatch({ type: 'ZOOM_OUT' });
        }
        break;
      case 'ArrowLeft':
        e.preventDefault();
        dispatch({ type: 'ROTATE_LEFT' });
        break;
      case 'ArrowRight':
        e.preventDefault();
        dispatch({ type: 'ROTATE_RIGHT' });
        break;
      case '0':
        e.preventDefault();
        dispatch({ type: 'RESET_VIEW' });
        break;
      }
    }
  }, [dispatch, viewport.zoomLevel]);

  return (
    <div
      ref={mapContainerRef}
      role="application"
      aria-label={`PDF map editor canvas, page ${currentPage + 1} of ${pdfPages.length}. Use Ctrl plus Plus to zoom in, Ctrl plus Minus to zoom out, Ctrl plus Arrow keys to rotate.`}
      tabIndex={0}
      onKeyDown={handleKeyDown}
      style={{
        ...MAP_CONTAINER_STYLES.BASE,
        cursor: getCursorStyle(viewport.isPanning, viewport.zoomLevel),
      }}
      onDrop={onDrop}
      onDragOver={onDragOver}
      onDragEnter={(e) => e.preventDefault()}
      onWheel={onWheel}
      onMouseDown={onMouseDown}
    >
      <ViewportControls
        state={{ viewport, imageDimensions }}
        dispatch={dispatch}
        containerRef={containerRef}
      />

      {pdfPages[currentPage] && (
        <div
          ref={containerRef}
          style={{
            ...MAP_CONTAINER_STYLES.VIEWPORT,
            transform: `translate(${viewport.panOffset.x}px, ${viewport.panOffset.y}px) scale(${viewport.zoomLevel})`,
            transition: getTransition(viewport.isPanning),
          }}
        >
          <img
            ref={mapImageRef}
            src={src ?? pdfPages[currentPage]}
            alt={`PDF Page ${currentPage + 1}`}
            style={{
              ...MAP_CONTAINER_STYLES.IMAGE,
              transform: `rotate(${viewport.rotation}deg)`,
            }}
            draggable={false}
            onLoad={onImageLoad}
          />

          {/* Access Points Layer */}
          <AccessPointLayer
            accessPointTypes={accessPointTypes}
            rotation={viewport.rotation}
            currentPageAccessPoints={currentPageAccessPoints}
            editingNodeId={editingNodeId}
            allowNodeMovement={allowNodeMovement}
            zoomLevel={viewport.zoomLevel}
            handleAccessPointDragStart={onAccessPointDragStart}
            startEditingNode={startEditingNode}
            showNodeDetails={showNodeDetails}
            handleTouchStartAccessPoint={onTouchStartAccessPoint}
          />
        </div>
      )}

      {/* Touch drag preview for mobile */}
      {touchDragPreview.visible && touchDragItem && (
        <div
          style={{
            ...TOUCH_DRAG_STYLES.CONTAINER,
            left: `${touchDragPreview.x}px`,
            top: `${touchDragPreview.y}px`,
          }}
        >
          <div
            style={{
              ...TOUCH_DRAG_STYLES.MARKER,
              backgroundColor: touchDragItem.isNew
                ? accessPointTypes.find(t => t.id === touchDragItem.type)?.color || COLORS.DEFAULT_AP_COLOR
                : accessPointTypes.find(t => t.id === touchDragItem.type)?.color || COLORS.DEFAULT_AP_COLOR,
            }}
          >
          </div>
          <div style={TOUCH_DRAG_STYLES.LABEL}>
            {touchDragItem.isNew
              ? `New ${accessPointTypes.find(t => t.id === touchDragItem.type)?.name || 'Access Point'}`
              : touchDragItem.name || 'Access Point'}
          </div>
        </div>
      )}
    </div>
  );
};
