/**
 * MainProjectSetupScreen Component
 * Main workspace for placing access points on the PDF map
 */

import React from 'react';
import {
  EuiFlexGroup,
  EuiFlexItem,
  EuiPanel,
  EuiSpacer,
  EuiText,
  EuiEmptyPrompt,
  EuiIcon,
  EuiLoadingSpinner,
  EuiButton
} from '@elastic/eui';
import { m } from 'framer-motion';
import { MapCanvas, ProjectControls } from '@/app/components/pdf-editor';
import { AccessPointTypes } from '@/features/access-points/components/AccessPointTypes';
import { CurrentPageAccessPoints } from '@/features/access-points/components/AccessPointList';
import { ErrorBoundary } from '@/shared/components';
import getDict from '@/lib/dict';
import { animations } from '@/features/project-setup/utils/animations';
import { isTypeAtLimit } from '@/lib/nodeScaling';
import type {
  ViewportState,
  ViewportAction,
  ImageDimensions,
  DragItem,
  TouchDragPreview,
  AccessPointMarkerPoint,
  AccessPointType,
  Project
} from '@/shared/types';

const dict = getDict("de");

interface MainProjectSetupScreenProps {
  pdfPages: string[];
  accessPointTypes: AccessPointType[];
  accessPoints: AccessPointMarkerPoint[];
  currentPageAccessPoints: AccessPointMarkerPoint[];
  projectFound: boolean;
  projectUpdated: boolean;
  allowNodeMovement: boolean;
  editingNodeId: string | null;
  editingNodeName: string;
  draggedItem: DragItem | null;
  touchDragPreview: TouchDragPreview;
  touchDragItem: DragItem | null;
  loading: boolean;
  viewport: ViewportState;
  imageDimensions: ImageDimensions;
  src: string;
  savedProjects: Project[];
  handleDragStart: (e: React.DragEvent<HTMLDivElement>, type: string) => void;
  handleTouchStartType: (e: React.TouchEvent<HTMLDivElement>, type: string) => void;
  handleAccessPointDragStart: (e: React.DragEvent<HTMLDivElement>, ap: AccessPointMarkerPoint) => void;
  handleTouchStartAccessPoint: (e: React.TouchEvent<HTMLDivElement>, ap: AccessPointMarkerPoint) => void;
  setEditingNodeName: (name: string) => void;
  startEditingNode: (ap: AccessPointMarkerPoint) => void;
  setAccessPoints: React.Dispatch<React.SetStateAction<AccessPointMarkerPoint[]>>;
  setEditingNodeId: (id: string | null) => void;
  onSave: () => void;
  onToggleNodeMovement: (value: boolean) => void;
  handleImageLoad: () => void;
  handleDrop: (e: React.DragEvent<HTMLDivElement>) => void;
  handleDragOver: (e: React.DragEvent<HTMLDivElement>) => void;
  handleWheel: (e: React.WheelEvent<HTMLDivElement>) => void;
  handleMouseDown: (e: React.MouseEvent<HTMLDivElement>) => void;
  showNodeDetails: (ap: AccessPointMarkerPoint) => void;
  dispatch: React.Dispatch<ViewportAction>;
  containerRef: React.RefObject<HTMLDivElement>;
  mapContainerRef: React.RefObject<HTMLDivElement>;
  mapImageRef: React.RefObject<HTMLImageElement>;
  mainRef: React.RefObject<HTMLDivElement>;
  onContinue: () => void;
}

export function MainProjectSetupScreen({
  pdfPages,
  accessPointTypes,
  accessPoints,
  currentPageAccessPoints,
  projectFound,
  projectUpdated,
  allowNodeMovement,
  editingNodeId,
  editingNodeName,
  draggedItem,
  touchDragPreview,
  touchDragItem,
  loading,
  viewport,
  imageDimensions,
  src,
  savedProjects,
  handleDragStart,
  handleTouchStartType,
  handleAccessPointDragStart,
  handleTouchStartAccessPoint,
  setEditingNodeName,
  startEditingNode,
  setAccessPoints,
  setEditingNodeId,
  onSave,
  onToggleNodeMovement,
  handleImageLoad,
  handleDrop,
  handleDragOver,
  handleWheel,
  handleMouseDown,
  showNodeDetails,
  dispatch,
  containerRef,
  mapContainerRef,
  mapImageRef,
  mainRef,
  onContinue
}: MainProjectSetupScreenProps) {
  return (
    <EuiFlexGroup gutterSize="l">
      <EuiFlexItem>
        {/* Access Point Types */}
        <AccessPointTypes
          accessPointTypes={accessPointTypes}
          isTypeAtLimit={isTypeAtLimit}
          getTypeCount={(type: string, aps: AccessPointMarkerPoint[]) =>
            aps.filter(ap => ap.type === type).length
          }
          handleDragStart={handleDragStart}
          handleTouchStartType={handleTouchStartType}
          accessPoints={accessPoints}
        />

        {/* Current Access Points */}
        <CurrentPageAccessPoints
          currentPageAccessPoints={currentPageAccessPoints}
          accessPointTypes={accessPointTypes}
          editingNodeName={editingNodeName}
          editingNodeId={editingNodeId}
          setEditingNodeName={setEditingNodeName}
          startEditingNode={startEditingNode}
          setAccessPoints={setAccessPoints}
          setEditingNodeId={setEditingNodeId}
        />
      </EuiFlexItem>

      {/* Map Display */}
      <EuiFlexItem grow={4} style={{ overflow: 'hidden' }} ref={mainRef}>
        {pdfPages.length > 0 ? (
          <EuiPanel>
            <ProjectControls
              projectName={projectFound ? savedProjects[0]?.name : ''}
              projectFound={projectFound}
              projectUpdated={projectUpdated}
              allowNodeMovement={allowNodeMovement}
              onSave={onSave}
              onToggleNodeMovement={onToggleNodeMovement}
            />

            <EuiSpacer size="m" />

            <ErrorBoundary>
              <MapCanvas
                pdfPages={pdfPages}
                currentPage={0}
                accessPoints={currentPageAccessPoints}
                accessPointTypes={accessPointTypes}
                viewport={viewport}
                imageDimensions={imageDimensions}
                allowNodeMovement={allowNodeMovement}
                editingNodeId={editingNodeId}
                draggedItem={draggedItem}
                touchDragPreview={touchDragPreview}
                touchDragItem={touchDragItem}
                src={src}
                onImageLoad={handleImageLoad}
                onDrop={handleDrop}
                onDragOver={handleDragOver}
                onWheel={handleWheel}
                onMouseDown={handleMouseDown}
                onAccessPointDragStart={handleAccessPointDragStart}
                onTouchStartAccessPoint={handleTouchStartAccessPoint}
                startEditingNode={startEditingNode}
                showNodeDetails={showNodeDetails}
                dispatch={dispatch}
                containerRef={containerRef}
                mapContainerRef={mapContainerRef}
                mapImageRef={mapImageRef}
              />
            </ErrorBoundary>

            <EuiSpacer size="s" />
            <EuiText size="s" color="subdued">
              <strong>Controls:</strong>
              <div>• {dict.dragText}. </div>
              <div>• {allowNodeMovement ? dict.nodeMovementText2 : dict.nodeMovementText} </div>
              <div>• {viewport.zoomLevel > 1 ? dict.panText : dict.zoomText} </div>
              <div>• {dict.rigthClick} </div>
            </EuiText>

            {!projectFound && (
              <m.div variants={animations.itemVariants} style={{ textAlign: 'center' }}>
                <div style={{ display: 'flex', gap: '12px', justifyContent: 'center' }}>
                  {!loading ? (
                    <EuiButton fill color={'danger'} onClick={onContinue}>
                      {dict.continue}
                    </EuiButton>
                  ) : (
                    <EuiLoadingSpinner size="l" />
                  )}
                </div>
              </m.div>
            )}
          </EuiPanel>
        ) : (
          <EuiEmptyPrompt
            icon={<EuiIcon type="document" size="xxl" />}
            title={
              <h2>
                <EuiLoadingSpinner />
              </h2>
            }
            body={<p>{dict.wait}</p>}
          />
        )}
      </EuiFlexItem>
    </EuiFlexGroup>
  );
}
