"use client"

// React Dependencies
import React from "react";
// NextJs Dependencies
import { useRouter } from 'next/navigation';
// External Libraries Dependencies
import { EuiBadge, EuiBasicTableColumn, EuiHealth, EuiFlexGroup, EuiFlexItem, EuiLink } from "@elastic/eui";
// Own Dependencies
import Device from "@/app/types/Device";
import Icon from "@/app/components/Icon";
import { CheckQuality}  from "@/app/components/QualityCheck";
import { TrafficConversion } from "@/app/components/TrafficConversion";
import getDict from "@/lib/dict";

const dict = getDict("de");
export const DeviceColumns = (showNetworkColumn: boolean) => {
  const router = useRouter();
  const [items, setItems] = React.useState<EuiBasicTableColumn<Device>[]>([]);

  React.useEffect(() => {
    const baseColumns: EuiBasicTableColumn<Device>[] = [
      {
        field: 'type',
        name: dict.type,
        sortable: true,
        truncateText: false,
        render: (item: string, device: Device) => (
          <EuiFlexGroup gutterSize="none" alignItems="center">
            <EuiFlexItem grow={false}>
              <EuiHealth color={device.online ? 'success' : 'danger'}>
                <Icon name={device.type} />
              </EuiHealth>
            </EuiFlexItem>
          </EuiFlexGroup>
        ),
      },
      {
        field: 'name',
        name: dict.name,
        sortable: true,
        truncateText: false,
        render: (name: string, device: Device) => (
          <span style={{ display: 'flex', alignItems: 'center', gap: '8px', whiteSpace: 'normal', wordBreak: 'break-word' }}>
            <CheckQuality value={device.signal} /> {name}
          </span>
        ),
      },
      {
        field: 'packets_rx',
        name: 'Packets /s',
        render: (item: number, device: Device) => (
          <div style={{ textAlign: 'center', fontSize: '13px' }}>
            <div style={{ color: '#07C' }}>↓ {TrafficConversion(device.packets_rx, '')}</div>
            <div style={{ color: '#BD271E' }}>↑ {TrafficConversion(device.packets_tx, '')}</div>
          </div>
        ),
      },
      {
        field: 'bytes_rx',
        name: 'Volume (MB)',
        render: (item: number, device: Device) => (
          <div style={{ textAlign: 'center', fontSize: '13px' }}>
            <div style={{ color: '#07C' }}>↓ {TrafficConversion(device.bytes_rx, '(MB)')}</div>
            <div style={{ color: '#BD271E' }}>↑ {TrafficConversion(device.bytes_tx, '(MB)')}</div>
          </div>
        ),
      },
      {
        field: 'rate_rx',
        name: 'Rate (MB/s)',
        render: (item: number, device: Device) => (
          <div style={{ textAlign: 'center', fontSize: '13px' }}>
            <div style={{ color: '#07C' }}>↓ {TrafficConversion(device.rate_rx, '(MB/s)')}</div>
            <div style={{ color: '#BD271E' }}>↑ {TrafficConversion(device.rate_tx, '(MB/s)')}</div>
          </div>
        ),
      },
      {
        field: 'accessPointName',
        name: dict.accessPoint,
        sortable: true,
        truncateText: false,
        render: (item: string, obj: Device) => obj.appName ?? '-',
      },
      {
        field: 'lastConnectedAt',
        name: dict.lastConnectedAt,
        sortable: true,
        render: (formatedDate: string, device: Device) => {
          const dateString = device.lastConnectedAt;

          // if empty
          if (!dateString) {
            return <EuiBadge color="hollow">{dict.unregistered}</EuiBadge>;
          }

          const date = new Date(dateString);

          const formattedDate = date.toLocaleDateString('de-DE', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
          });

          const formattedTime = date.toLocaleTimeString('de-DE', {
            hour: '2-digit',
            minute: '2-digit',
          });

          return (
            <span style={{ whiteSpace: 'nowrap' }}>
              {formattedDate}<br />
              <small style={{ color: '#69707d' }}>{formattedTime}</small>
            </span>
          );
        },
      },
      {
        name: dict.actions,
        actions: [
          {
            name: 'Edit',
            description: 'Edit device',
            icon: 'pencil',
            type: 'icon',
            onClick: (device: Device) => {
              router.push(`/devices/edit/?id=${device.id}`);
            },
          },
        ],
      },
    ];

    if (showNetworkColumn) {
      const networkColumn = {
        field: 'network',
        name: dict.network,
        width: '150px',
        truncateText: false,
        render: (item: string, obj: Device) => {
          if (obj.networkName !== undefined) {
            return (<EuiLink href={`/networks/edit/?id=${obj.network}`} external>{obj.networkName} </EuiLink>);
          }
          return  '-';
        },
      };

      baseColumns.splice(5, 0, networkColumn);
    }

    setItems(baseColumns);
  }, [showNetworkColumn, router]);

  return items;
};
