"use client"

// React Dependencies
import React from "react";
// Custum hooks
import { useHighlightedItems, useFilteredData } from "@/shared/hooks";
// External Libraries Dependencies
import {
  EuiInMemoryTable,
  EuiSelectOption,
  EuiFlexGroup,
  EuiFlexItem,
  EuiText,
  EuiSpacer,
} from "@elastic/eui";
// Own Dependencies
import { DeviceColumns } from "./deviceColumns";
import Device, { DeviceType } from "@/app/types/Device";
import Filters from "@/app/components/Filters";
import getDict from "@/lib/dict";
const dict = getDict("de");
interface SortConfig {
  field: keyof Device;
  direction: "asc" | "desc";
}


export const DeviceTable = (props: { data: Device[], showNetworkColumn: boolean}) => {
  const column = DeviceColumns(props.showNetworkColumn);
  const [sortConfig, setSortConfig] = React.useState<SortConfig>({
    field: 'name',
    direction: "asc",
  });

  const sorting: { sort: SortConfig } = {
    sort: sortConfig,
  };
  const { items, newRowIds } = useHighlightedItems(props.data);
  const { filteredItems, searchText, setSearchText, selectedType, setSelectedType, onlineFilter, setOnlineFilter } =
    useFilteredData(items, true);
  const typeOptions: EuiSelectOption[] = [
    {
      value: '',
      text: 'All'
    },
    ...Array.from(new Set(items.map((item: Device) => item.type))).map((type: DeviceType) => ({
      value: type,
      text: type.charAt(0).toUpperCase() + type.slice(1),
    })),
  ];
  const rowProps = (item: { id: string | number }) => {
    const isNew = newRowIds.includes(item.id);
    return {
      className: isNew ? 'new-row-highlight' : '',
    };
  };

  // Pagination configuration
  const [pageIndex, setPageIndex] = React.useState(0);
  const [pageSize, setPageSize] = React.useState(10);

  const pagination = {
    pageIndex,
    pageSize,
    totalItemCount: filteredItems.length,
    pageSizeOptions: [10, 25, 50, 100],
    showPerPageOptions: true,
  };

  const onTableChangeWithPagination = (criteria: any) => {
    if (criteria.page) {
      const { index, size } = criteria.page;
      setPageIndex(index);
      setPageSize(size);
    }
    if (criteria.sort) {
      setSortConfig(criteria.sort);
    }
  };

  return(
    <>
      {/* Filter Section */}
      <Filters
        searchText={searchText}
        setSearchText={setSearchText}
        selectedType={selectedType}
        setSelectedType={setSelectedType}
        onlineFilter={onlineFilter}
        setOnlineFilter={setOnlineFilter}
        typeOptions={typeOptions}
        showOnline={true}
      />

      <EuiSpacer size="m" />

      {/* Results Count */}
      <EuiFlexGroup justifyContent="spaceBetween" alignItems="center">
        <EuiFlexItem grow={false}>
          <EuiText size="s" color="subdued">
            {dict.show} <strong>{filteredItems.length}</strong> von <strong>{items.length}</strong> {dict.devices}
          </EuiText>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="s" />

      {/* Table */}
      <EuiInMemoryTable
        items={filteredItems as Device[]}
        columns={column}
        onTableChange={onTableChangeWithPagination}
        sorting={sorting}
        pagination={pagination}
        rowProps={rowProps}
        tableLayout="auto"
        compressed={false}
      />
    </>
  )
}
