"use client"
// React Dependencies
import React, { Suspense, useRef } from "react";
// NextJs Dependencies
import { useRouter, useSearchParams } from "next/navigation";
// API Client
import { api } from "@/lib/api";
// External Libraries Dependencies
import { useReactToPrint } from "react-to-print";
import QRCode from "react-qr-code";
import {
  EuiDescriptionList,
  EuiTitle,
  EuiConfirmModal,
  useGeneratedHtmlId,
  EuiHealth,
  EuiButton,
  EuiPanel,
  EuiFlexGroup,
  EuiFlexItem,
  EuiSpacer,
  EuiButtonEmpty,
  EuiBreadcrumbs,
  EuiText,
  EuiStat,
} from "@elastic/eui";
import { Printer } from 'lucide-react';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import Icon from "@/app/components/Icon";
import { useToast } from "@/app/components/ToastProvider";
import { CheckQuality } from "@/app/components/QualityCheck";
import { TrafficConversion } from "@/app/components/TrafficConversion";
// Own Dependencies
import getDict from "@/lib/dict";
import Device from "@/app/types/Device";
import Network from "@/app/types/Network";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { formatDateTime } from "@/shared/utils";

import styles from './page.module.css';
import { PrintComponent } from "./print";

function DeviceDetailsContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const id = searchParams.get("id");
  const dict = getDict("de");
  const [device, setDevice] = React.useState<Device | null>(null);
  const [network, setNetwork] = React.useState<Network | null>(null);
  const { addToast } = useToast();

  React.useEffect(() => {

    async function fetchDevice() {
      if (!id) return;

      try {
        const data = await api.devices.getDevice(id);
        setDevice(data);
        if (data.network) fetchNetwork(data.network);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.deviceFetchToast} ${id}`,
        });
        router.push("/devices");
      }
    }

    async function fetchNetwork(id: string) {
      if (!id) return;

      try {
        const data = await api.networks.getNetwork(id);
        setNetwork(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail} ${id}`,
        });
      }
    }

    const interval = setInterval(() => {
      fetchDevice();
    }, 5000);

    fetchDevice();

    return () => clearInterval(interval);
  }, [id, router]);

  async function removeDevice() {
    if (!id) return;

    try {
      await api.devices.deleteDevice(id);
      setIsDestroyModalVisible(false);
      addToast({
        title: "Success",
        color: "success",
        text: `${dict.deviceDeleted}`,
      });
      router.push("/devices");
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: `${dict.deviceFetchToastFail} ${id}`,
      });
    }
  }
  const items = device?.type ? ([
    {
      title: dict.type,
      description:
        <>
          <EuiHealth color={device.online ? 'green' : 'danger'}>
            <Icon name={device.type} /></EuiHealth>
        </>
    },
    {
      title: dict.signal,
      description: <CheckQuality value={device.signal} />,
    },
    {
      title: dict.name,
      description: device.name ?? "",
    },
    {
      title: dict.up,
      description: TrafficConversion(device.bytes_rx, 'sortUp') ?? "",
    },
    {
      title: dict.down,
      description: TrafficConversion(device.bytes_tx, 'sortDown') ?? "",
    },
    {
      title: dict.network,
      description: network?.name ?? "",
    },
    // {
    //   title: dict.addedBy,
    //   description: device.addedBy ?? "",
    // },
    {
      title: dict.addedAt,
      description: formatDateTime(device.addedAt)
    },
    {
      title: dict.password,
      description: device.password ?? "",
    },
    {
      title: dict.password + ' QR-Code:',
      description: <>{device.password && network ?
        <QRCode
          value={`WIFI:S:${network.name};T:WPA;P:${device.password};H:false;`}
          style={{maxWidth: "10rem", height: "auto"}}
        />
        : ""}</>,
    },
    {
      title: dict.firstConnectedAt,
      description: formatDateTime(device.firstConnectedAt)
    },
    {
      title: dict.lastConnectedAt,
      description: formatDateTime(device.lastConnectedAt),
    },
    {
      title: dict.macAddress,
      description: device.mac || "N/A",
    },
    // {
    //   title: dict.ipAddress,
    //   description: device.ipAddress ?? "",
    // },
  ]) : [];
  const [isDestroyModalVisible, setIsDestroyModalVisible] = React.useState(false);
  const destroyModalTitleId = useGeneratedHtmlId();
  const closeDestroyModal = () => setIsDestroyModalVisible(false);
  const showDestroyModal = () => setIsDestroyModalVisible(true);

  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  // Breadcrumbs
  const breadcrumbs = [
    {
      text: 'Home',
      href: '/',
    },
    {
      text: 'Devices',
      href: '/devices',
    },
    {
      text: device?.name || 'Loading...',
    },
  ];

  // Loading state
  if (!device) {
    return (
      <div>
        <EuiBreadcrumbs breadcrumbs={breadcrumbs} />
        <EuiSpacer size="m" />
        <EuiTitle>
          <h2>{dict.editDevice}</h2>
        </EuiTitle>
        <EuiSpacer size="l" />
      </div>
    );
  }

  // Separate items into categories
  const basicInfo = [
    {
      title: dict.name,
      description: device.name ?? "",
    },
    {
      title: dict.type,
      description: (
        <EuiHealth color={device.online ? 'success' : 'danger'}>
          <Icon name={device.type} />
        </EuiHealth>
      ),
    },
    {
      title: dict.network,
      description: network?.name ?? "",
    },
    {
      title: dict.macAddress,
      description: device.mac || "N/A",
    },
  ];

  const performanceInfo = [
    {
      title: dict.signal,
      description: <CheckQuality value={device.signal} />,
    },
    {
      title: dict.up,
      description: TrafficConversion(device.bytes_rx, 'sortUp') ?? "",
    },
    {
      title: dict.down,
      description: TrafficConversion(device.bytes_tx, 'sortDown') ?? "",
    },
  ];

  const connectionInfo = [
    {
      title: dict.addedAt,
      description: formatDateTime(device.addedAt)
    },
    {
      title: dict.firstConnectedAt,
      description: formatDateTime(device.firstConnectedAt)
    },
    {
      title: dict.lastConnectedAt,
      description: formatDateTime(device.lastConnectedAt),
    },
  ];

  const securityInfo = device.password ? [
    {
      title: dict.password,
      description: device.password,
    },
  ] : [];

  return (
    <div>
      {/* Header Section */}
      <EuiBreadcrumbs breadcrumbs={breadcrumbs} />
      <EuiSpacer size="m" />

      <EuiFlexGroup justifyContent="spaceBetween" alignItems="center">
        <EuiFlexItem grow={false}>
          <div>
            <EuiTitle size="l">
              <h1>{device.name}</h1>
            </EuiTitle>
            <EuiSpacer size="s" />
            <EuiHealth color={device.online ? 'success' : 'danger'}>
              {device.online ? 'Online' : 'Offline'}
            </EuiHealth>
          </div>
        </EuiFlexItem>
        <EuiFlexItem grow={false}>
          <EuiFlexGroup gutterSize="s">
            <EuiFlexItem grow={false}>
              <EuiButtonEmpty
                iconType={() => <Printer size={16} />}
                onClick={() => reactToPrintFn()}
              >
                Print
              </EuiButtonEmpty>
            </EuiFlexItem>
            <EuiFlexItem grow={false}>
              <EuiButton
                onClick={showDestroyModal}
                color="danger"
                fill
              >
                {dict.remove}
              </EuiButton>
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="l" />
      {/* Print-only content */}
      <div ref={contentRef} className={styles.printOnly}>
        <PrintComponent data={items} />
      </div>

      {/* Stats Overview */}
      <EuiFlexGroup>
        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiStat
              title={TrafficConversion(device.bytes_rx, 'sortUp') ?? "0 B"}
              description={dict.up}
              titleColor="success"
              textAlign="center"
            />
          </EuiPanel>
        </EuiFlexItem>
        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiStat
              title={TrafficConversion(device.bytes_tx, 'sortDown') ?? "0 B"}
              description={dict.down}
              titleColor="primary"
              textAlign="center"
            />
          </EuiPanel>
        </EuiFlexItem>
        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiStat
              title={<CheckQuality value={device.signal} />}
              description={dict.signal}
              textAlign="center"
            />
          </EuiPanel>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="l" />

      {/* Device Information Panels */}
      <EuiFlexGroup>
        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiTitle size="s">
              <h3>Basic Information</h3>
            </EuiTitle>
            <EuiSpacer size="m" />
            <EuiDescriptionList
              listItems={basicInfo}
              type="column"
              compressed
            />
          </EuiPanel>
        </EuiFlexItem>

        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiTitle size="s">
              <h3>Performance</h3>
            </EuiTitle>
            <EuiSpacer size="m" />
            <EuiDescriptionList
              listItems={performanceInfo}
              type="column"
              compressed
            />
          </EuiPanel>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="m" />

      <EuiFlexGroup>
        <EuiFlexItem>
          <EuiPanel hasBorder>
            <EuiTitle size="s">
              <h3>Connection History</h3>
            </EuiTitle>
            <EuiSpacer size="m" />
            <EuiDescriptionList
              listItems={connectionInfo}
              type="column"
              compressed
            />
          </EuiPanel>
        </EuiFlexItem>

        {device.password && network && (
          <EuiFlexItem>
            <EuiPanel hasBorder>
              <EuiTitle size="s">
                <h3>Network Credentials</h3>
              </EuiTitle>
              <EuiSpacer size="m" />
              <EuiDescriptionList
                listItems={securityInfo}
                type="column"
                compressed
              />
              <EuiSpacer size="m" />
              <EuiText size="s">
                <strong>QR Code:</strong>
              </EuiText>
              <EuiSpacer size="s" />
              <div style={{ display: 'flex', justifyContent: 'center' }}>
                <QRCode
                  value={`WIFI:S:${network.name};T:WPA;P:${device.password};H:false;`}
                  style={{ maxWidth: "10rem", height: "auto" }}
                />
              </div>
            </EuiPanel>
          </EuiFlexItem>
        )}
      </EuiFlexGroup>

      {/* Delete Confirmation Modal */}
      {isDestroyModalVisible && (
        <EuiConfirmModal
          aria-labelledby={destroyModalTitleId}
          title={`${dict.remove} ${device?.name}?`}
          titleProps={{ id: destroyModalTitleId }}
          onCancel={closeDestroyModal}
          onConfirm={removeDevice}
          cancelButtonText={dict.cancel}
          confirmButtonText={dict.confirm}
          buttonColor="danger"
          defaultFocusedButton="confirm"
        >
          <EuiText>
            <p>This action cannot be undone. Are you sure you want to delete this device?</p>
          </EuiText>
        </EuiConfirmModal>
      )}
    </div>
  );
}

export default function DeviceDetails() {
  return (
    <AuthCheck>
      <Navigation>
        <Suspense fallback={<div>Loading...</div>}>
          <DeviceDetailsContent />
        </Suspense>
      </Navigation>
    </AuthCheck>
  );
}
