"use client"

import React from "react";
// Custom Hooks
import { useDomReady } from "@/shared/hooks";
import { useAuthContext } from "@/features/auth";
import { useDevices } from "@/features/devices";
import { useNetworks } from "@/features/networks";
// API Client
import { api } from "@/lib/api";
// Context
import { useLoader } from "@/shared/providers";
// External Libraries Dependencies
import {
  EuiPage,
  EuiPageBody,
  EuiTitle,
  EuiFlexItem,
  EuiButton,
  EuiFlexGroup,
  EuiLoadingSpinner,
  EuiSwitch,
  EuiSpacer,
  EuiPanel,
  EuiIcon,
  EuiStat,
  EuiBreadcrumbs,
} from '@elastic/eui';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import { useToast } from "@/app/components/ToastProvider";
import { AddModal } from "@/app/components/AddModal";
import { DeviceTable } from "@/app/devices/deviceTable";
// Own Dependencies
import getDict from "@/lib/dict";
import Device  from "../types/Device";
import Network from "../types/Network";
import AP from "../types/AP";
import {bytesToMB} from "@/shared/utils";


const dict = getDict("de");

export default function Devices() {
  // State
  const [sortedData, setSortedData] = React.useState<Device[]>([]);
  const [auto, setAuto] = React.useState(true);
  const [isModalVisible, setIsModalVisible] = React.useState(false);

  // Hooks
  const { addToast } = useToast();
  const { setLoading } = useLoader();
  const domReady = useDomReady();
  const { isAuthenticated } = useAuthContext();

  // Use custom hooks for data fetching
  const {
    devices,
    loading: devicesLoading,
    error: devicesError
  } = useDevices({
    autoFetch: domReady && isAuthenticated,
    refetchInterval: auto ? 5000 : 0,
  });

  const {
    networks,
    loading: networksLoading,
    error: networksError
  } = useNetworks({
    autoFetch: domReady && isAuthenticated,
    refetchInterval: auto ? 5000 : 0,
  });

  // Fetch access points (not yet migrated to hook)
  const [accessPoints, setAccessPoints] = React.useState<AP[]>([]);
  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;

    const fetchAPs = async () => {
      try {
        const data = await api.accessPoints.getAccessPoints();
        setAccessPoints(data || []);
      } catch (err) {
        console.error('Failed to fetch access points:', err);
      }
    };

    fetchAPs();
    const interval = auto ? setInterval(fetchAPs, 5000) : null;

    return () => {
      if (interval) clearInterval(interval);
    };
  }, [auto, domReady, isAuthenticated]);

  // Combine loading states
  const loading = devicesLoading || networksLoading;

  // Handle loading state
  React.useEffect(() => {
    setLoading(loading);
  }, [loading, setLoading]);

  // Handle errors with toast
  React.useEffect(() => {
    if (devicesError) {
      addToast({
        title: "Error",
        color: "danger",
        text: `${dict.deviceFetchToastFail}`,
      });
    }
    if (networksError) {
      addToast({
        title: "Error",
        color: "danger",
        text: `Failed to fetch networks`,
      });
    }
  }, [devicesError, networksError, addToast]);

  // Enrich devices with network and AP names
  React.useEffect(() => {
    if (devices.length > 0 && networks.length > 0) {
      const enrichedDevices = devices.map((device: Device) => ({
        ...device,
        networkName: networks.find((item: Network) => item.id === device.network)?.name || 'Unknown',
        appName: device.currentAP && accessPoints.length > 0
          ? accessPoints.find((item: AP) => item.id === device.currentAP)?.name || '-'
          : '-',
      }));
      setSortedData(enrichedDevices);
    }
  }, [devices, networks, accessPoints]);

  // Modal handlers
  const showModal = () => setIsModalVisible(true);
  const closeModal = () => setIsModalVisible(false);

  // Calculate device statistics
  const totalDevices = sortedData.length;
  const onlineDevices = sortedData.filter((d) => d.online).length;
  const offlineDevices = totalDevices - onlineDevices;
  const totalTrafficDown = bytesToMB(sortedData.reduce((sum, d) => sum + (d.bytes_rx || 0), 0));
  const totalTrafficUp = bytesToMB(sortedData.reduce((sum, d) => sum + (d.bytes_tx || 0), 0));

  // Breadcrumbs
  const breadcrumbs = [
    {
      text: 'Home',
      href: '/',
    },
    {
      text: dict.devices,
    },
  ];

  // Loading state
  if (!domReady) {
    return (
      <EuiPage paddingSize="none">
        <EuiPageBody>
          <EuiLoadingSpinner size="xl" />
        </EuiPageBody>
      </EuiPage>
    );
  }
  return (
    <AuthCheck>
      <Navigation>
        <div>
          {/* Breadcrumbs */}
          <EuiBreadcrumbs breadcrumbs={breadcrumbs} />
          <EuiSpacer size="m" />

          {/* Page Header */}
          <EuiFlexGroup justifyContent="spaceBetween" alignItems="center">
            <EuiFlexItem grow={false}>
              <EuiFlexGroup alignItems="center" gutterSize="s">
                <EuiFlexItem grow={false}>
                  <EuiIcon type="compute" size="xl" />
                </EuiFlexItem>
                <EuiFlexItem grow={false}>
                  <EuiTitle size="l">
                    <h1>{dict.devices}</h1>
                  </EuiTitle>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiFlexItem>
            <EuiFlexItem grow={false}>
              <EuiFlexGroup alignItems="center" gutterSize="m">
                <EuiFlexItem grow={false}>
                  <EuiButton
                    color="success"
                    iconType="plusInCircleFilled"
                    onClick={showModal}
                    fill
                  >
                    {dict.addDevice}
                  </EuiButton>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiFlexItem>
          </EuiFlexGroup>

          <EuiSpacer size="l" />
          <EuiFlexGroup>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiStat
                  title={totalDevices.toString()}
                  description={dict.devices || "Total Devices"}
                  titleColor="primary"
                  textAlign="center"
                >
                  <EuiIcon type="compute" size="l" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiStat
                  title={onlineDevices.toString()}
                  description="Online"
                  titleColor="success"
                  textAlign="center"
                >
                  <EuiIcon type="checkInCircleFilled" size="l" color="success" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiStat
                  title={offlineDevices.toString()}
                  description="Offline"
                  titleColor="danger"
                  textAlign="center"
                >
                  <EuiIcon type="minusInCircleFilled" size="l" color="danger" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiStat
                  title={`${totalTrafficDown} MB`}
                  description={dict.down || "Volumen ↑"}
                  titleColor="success"
                  textAlign="center"
                >
                  <EuiIcon type="sortDown" size="l" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
            <EuiFlexItem>
              <EuiPanel hasBorder>
                <EuiStat
                  title={`${totalTrafficUp} MB`}
                  description={dict.up || "Volumen ↓"}
                  titleColor="primary"
                  textAlign="center"
                >
                  <EuiIcon type="sortUp" size="l" />
                </EuiStat>
              </EuiPanel>
            </EuiFlexItem>
          </EuiFlexGroup>

          <EuiSpacer size="l" />

          {/* Auto-refresh Toggle */}
          <EuiPanel>
            <EuiSwitch
              label={dict.refreshData}
              checked={auto}
              onChange={() => {
                setAuto(!auto);
                addToast({
                  title: auto ? dict.refreshDataDisabled : dict.refreshDataEnabled,
                  color: "primary",
                });
              }}
            />
          </EuiPanel>

          <EuiSpacer size="l" />

          {/* Devices Table */}
          <EuiPanel>
            <DeviceTable data={sortedData} showNetworkColumn={true} />
          </EuiPanel>

          {/* Add Device Modal */}
          {isModalVisible && <AddModal closeModal={closeModal} />}
        </div>
      </Navigation>
    </AuthCheck>
  );
}
