"use client"

/**
 * Internet Statistics Page
 * Professional UI showing network traffic and device statistics
 *
 * Improvements:
 * - Consistent layout with EuiPage structure
 * - Professional header with icon
 * - Better spacing and visual hierarchy
 * - EUI icons instead of Lucide
 * - Refresh button
 * - Auto-refresh toggle
 * - Better error handling
 * - Loading states
 */

import React from "react";
// Custom hooks
import { useAuthContext } from "@/features/auth";
import { useDomReady } from "@/shared/hooks";
// API Client
import { api } from "@/lib/api";
// External Libraries Dependencies
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiPageHeaderSection,
  EuiTitle,
  EuiFlexGroup,
  EuiFlexItem,
  EuiPanel,
  EuiStat,
  EuiLoadingSpinner,
  EuiIcon,
  EuiButton,
  EuiSpacer,
  EuiSwitch,
} from '@elastic/eui';
// Components
import Navigation from "@/app/components/navigation/Navigation";
import AuthCheck from "@/app/components/AuthCheck";
import { useToast } from "@/app/components/ToastProvider";
import { useLoader } from "@/shared/providers";
// Own Dependencies
import Device from "@/app/types/Device";
import getDict from "@/lib/dict";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import {getUserStats, capitalize, bytesToMB} from "@/shared/utils";

export default function Internet() {
  const dict = getDict("de");

  // State
  const [auto, setAuto] = React.useState(true);
  const [traffic, setTraffic] = React.useState(0);
  const [up, setUp] = React.useState(0);
  const [down, setDown] = React.useState(0);
  const [devices, setDevices] = React.useState<Device[]>([]);

  // Hooks
  const domReady = useDomReady();
  const { isAuthenticated } = useAuthContext();
  const { addToast } = useToast();
  const { setLoading } = useLoader();

  const { onlineCount, totalCount } = getUserStats(devices);

  // Fetch devices and calculate stats
  const fetchDevice = React.useCallback(async () => {
    try {
      setLoading(true);
      const data = await api.devices.getDevices();
      const sumKb = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_rx ?? 0) + (obj.bytes_tx ?? 0), 0);
      const sumUp = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_tx ?? 0), 0);
      const sumDown = data.reduce((acc: number, obj: Device) => acc + (obj.bytes_rx ?? 0), 0);
      setTraffic(bytesToMB(sumKb));
      setUp(bytesToMB(sumUp));
      setDown(bytesToMB(sumDown));
      setDevices(data);
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('DEVICE_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: "danger",
        text: `${dict.deviceFetchToastFail}`,
      });
    } finally {
      setLoading(false);
    }
  }, [addToast, setLoading, dict.deviceFetchToastFail]);

  // Auto-fetch effect
  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;

    fetchDevice();

    if (auto) {
      const interval = setInterval(fetchDevice, 5000);
      return () => clearInterval(interval);
    }
  }, [auto, domReady, isAuthenticated, fetchDevice]);

  // Loading state
  if (!domReady) {
    return (
      <EuiPage paddingSize="none">
        <EuiPageBody>
          <EuiLoadingSpinner size="xl" />
        </EuiPageBody>
      </EuiPage>
    );
  }
  return (
    <AuthCheck>
      <Navigation>
        <EuiPage paddingSize="l">
          <EuiPageBody>
            {/* Page Header */}
            <EuiPageHeader>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="s">
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="cloudStormy" size="xl" />
                  </EuiFlexItem>
                  <EuiFlexItem grow={false}>
                    <EuiTitle size="l">
                      <h1>Internet Statistics</h1>
                    </EuiTitle>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="m">
                  <EuiFlexItem grow={false}>
                    <EuiButton
                      iconType="refresh"
                      onClick={fetchDevice}
                      aria-label="Refresh statistics"
                    >
                      Refresh
                    </EuiButton>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
            </EuiPageHeader>

            <EuiSpacer size="l" />

            {/* Auto-refresh Toggle */}
            <EuiPanel>
              <EuiSwitch
                label={dict.refreshData}
                checked={auto}
                onChange={() => {
                  setAuto(!auto);
                  addToast({
                    title: auto ? dict.refreshDataDisabled : dict.refreshDataEnabled,
                    color: "primary",
                  });
                }}
              />
            </EuiPanel>

            <EuiSpacer size="l" />

            {/* Network Traffic Statistics */}
            <EuiPanel>
              <EuiFlexGroup alignItems="center" gutterSize="s">
                <EuiFlexItem grow={false}>
                  <EuiIcon type="visBarVertical" size="m" color="primary" />
                </EuiFlexItem>
                <EuiFlexItem grow={false}>
                  <EuiTitle size="s">
                    <h3>Network Traffic</h3>
                  </EuiTitle>
                </EuiFlexItem>
              </EuiFlexGroup>

              <EuiSpacer size="m" />

              <EuiFlexGroup gutterSize="l" wrap>
                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="primary" hasShadow={false}>
                    <EuiStat
                      title={traffic.toFixed(1) + ' MB'}
                      description="Total Traffic"
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="storage" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>

                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="success" hasShadow={false}>
                    <EuiStat
                      title={down.toFixed(1) + ' MB'}
                      description={capitalize(dict.download)}
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="sortDown" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>

                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="warning" hasShadow={false}>
                    <EuiStat
                      title={up.toFixed(1) + ' MB'}
                      description={capitalize(dict.upload)}
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="sortUp" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiPanel>

            <EuiSpacer size="l" />

            {/* Device Statistics */}
            <EuiPanel>
              <EuiFlexGroup alignItems="center" gutterSize="s">
                <EuiFlexItem grow={false}>
                  <EuiIcon type="users" size="m" color="accent" />
                </EuiFlexItem>
                <EuiFlexItem grow={false}>
                  <EuiTitle size="s">
                    <h3>{capitalize(dict.clientStats)}</h3>
                  </EuiTitle>
                </EuiFlexItem>
              </EuiFlexGroup>

              <EuiSpacer size="m" />

              <EuiFlexGroup gutterSize="l" wrap>
                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="accent" hasShadow={false}>
                    <EuiStat
                      title={totalCount}
                      description="Total Devices"
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="compute" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>

                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="success" hasShadow={false}>
                    <EuiStat
                      title={onlineCount}
                      titleColor="success"
                      description={capitalize(dict.online)}
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="check" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>

                <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
                  <EuiPanel color="danger" hasShadow={false}>
                    <EuiStat
                      title={totalCount - onlineCount}
                      titleColor="danger"
                      description={capitalize(dict.offline)}
                      titleSize="s"
                      textAlign="center"
                    >
                      <EuiIcon type="cross" size="xl" />
                    </EuiStat>
                  </EuiPanel>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiPanel>
          </EuiPageBody>
        </EuiPage>
      </Navigation>
    </AuthCheck>
  );
}
