"use client"
// React Dependencies
import React from 'react';
// NextJs Dependencies
import { useRouter } from "next/navigation";
import Image from "next/image";
// External Libraries Dependencies
import {
  EuiButton,
  EuiFieldText,
  EuiLoadingSpinner,
  EuiFieldPassword,
  EuiText,
  EuiTitle,
  EuiSpacer,
  EuiCallOut,
  EuiIcon
} from "@elastic/eui";
// Own Dependencies
import styles from "./page.module.css";
import getDict from "@/lib/dict";
import branding from "@/lib/branding";
import { validateUsername } from "@/lib/security/validation";
import { logSecurityEvent, SecurityEventType } from "@/lib/security/auth";
import { useRateLimit } from "@/lib/security/rateLimit";
import { useAuthContext } from "@/features/auth";
import { useLoginSplash } from "@/shared/providers";
import { api } from "@/lib/api";
type LoginState = 'idle' | 'loading' | 'success' | 'error';

export default function Login() {
  const dict = getDict("de");
  const router = useRouter();
  const { login: authLogin, hideLogoutSplash } = useAuthContext();
  const { setShowLoginSplash, setLoginSplashFadeOut } = useLoginSplash();

  // Rate limiting (5 attempts per 15 minutes)
  const { record: recordAttempt } = useRateLimit(
    { maxAttempts: 5, windowMs: 15 * 60 * 1000 },
    'login'
  );

  // Form state
  const [username, setUsername] = React.useState('');
  const [password, setPassword] = React.useState('');

  // UI state
  const [loginState, setLoginState] = React.useState<LoginState>('idle');
  const [errorMessage, setErrorMessage] = React.useState('');
  const [usernameError, setUsernameError] = React.useState('');
  const [passwordError, setPasswordError] = React.useState('');

  // Validate username on blur
  const handleUsernameBlur = () => {
    if (username.length === 0) return;

    const validation = validateUsername(username);
    if (!validation.valid) {
      setUsernameError(validation.error || 'Invalid username');
    } else {
      setUsernameError('');
    }
  };

  // Validate password on blur
  const handlePasswordBlur = () => {
    if (password.length === 0) return;

    if (password.length < 6) {
      setPasswordError('Password must be at least 6 characters');
    } else {
      setPasswordError('');
    }
  };

  // Handle form submission
  async function handleSubmit(event?: React.FormEvent) {
    event?.preventDefault();

    // Clear previous errors
    setErrorMessage('');
    setUsernameError('');
    setPasswordError('');

    setLoginState('loading');

    try {
      const rememberMe = false;
      // Use new API client for login
      const response = await api.auth.login({
        username,
        password,
        rememberMe
      });

      // Check for successful login
      if (response && response.apiKey) {
        // Record successful attempt
        recordAttempt(true);

        // Log security event
        logSecurityEvent({
          type: SecurityEventType.LOGIN_SUCCESS,
          timestamp: Date.now(),
          metadata: { username },
        });

        // Update auth context (this will set cookies and auth state)
        authLogin(username, response.apiKey);

        setLoginState('success');

        // Show the splash screen immediately
        setShowLoginSplash(true);

        // Keep splash visible for at least 2 seconds before navigating
        setTimeout(() => {
          router.push("/");
          // Start fade out after navigation begins
          setTimeout(() => {
            setLoginSplashFadeOut(true);
          }, 1000);
          // Splash will be hidden by the home page when it mounts
        }, 2000);
      } else {
        // Record failed attempt
        recordAttempt(false);

        // Log security event
        logSecurityEvent({
          type: SecurityEventType.LOGIN_FAILED,
          timestamp: Date.now(),
          metadata: { username, reason: 'Invalid credentials' },
        });

        // Handle unsuccessful login response
        setLoginState('error');
        const errorMsg = response.error || dict.invalidCredentials;
        setErrorMessage(errorMsg);
      }
    } catch (error) {
      // Record failed attempt
      recordAttempt(false);

      setLoginState('error');

      const errorMsg = dict.loginError;
      setErrorMessage(errorMsg);

    }
  }

  // Hide logout splash when login page mounts
  React.useEffect(() => {
    // Small delay to ensure page has rendered before hiding splash
    const timer = setTimeout(() => {
      hideLogoutSplash();
    }, 100);

    return () => clearTimeout(timer);
  }, [hideLogoutSplash]);

  // Handle Enter key
  React.useEffect(() => {
    const listener = (event: KeyboardEvent) => {
      if (event.code === 'Enter' || event.code === 'NumpadEnter') {
        if (password.length >= 6 && username.length >= 4 && loginState === 'idle') {
          handleSubmit();
        }
      }
    };

    document.addEventListener('keydown', listener);
    return () => {
      document.removeEventListener('keydown', listener);
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [password, username, loginState]);

  const onUsernameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setUsername(value);

    // Real-time validation as user types
    if (value.length > 0) {
      const validation = validateUsername(value);
      if (!validation.valid) {
        setUsernameError(validation.error || 'Invalid username');
      } else {
        setUsernameError('');
      }
    } else {
      setUsernameError('');
    }
  };

  const onPasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setPassword(e.target.value);
    if (passwordError) setPasswordError('');
  };

  const isFormValid = username.length >= 0 && password.length >= 6 && !usernameError && !passwordError;
  const isLoading = loginState === 'loading';
  const isSuccess = loginState === 'success';
  return (
    <div className={styles.container}>
      <div className={styles.loginCard}>
        <form className={styles.form} onSubmit={handleSubmit}>
          {/* Branding */}
          <div className={styles.branding}>
            <div className={styles.logoWrapper}>
              <Image
                src="/logo.svg"
                alt="Logo"
                width={60}
                height={60}
                priority
                className={styles.logo}
              />
            </div>
            <div className={styles.brandingText}>
              <EuiTitle size="l">
                <h1>{branding.productName}</h1>
              </EuiTitle>
              <EuiText size="s" color="subdued">
                <p>{branding.companyName}</p>
              </EuiText>
            </div>
          </div>

          <EuiSpacer size="l" />

          {/* Title */}
          <div className={styles.titleSection}>
            <EuiTitle size="m">
              <h2>{dict.login}</h2>
            </EuiTitle>
            <EuiText size="s" color="subdued">
              <p>{dict.loginText}</p>
            </EuiText>
          </div>

          <EuiSpacer size="m" />

          {/* Error Message */}
          {loginState === 'error' && errorMessage && (
            <>
              <EuiCallOut
                title={dict.loginFailed}
                color="danger"
                iconType="alert"
                size="s"
              >
                <p>{errorMessage}</p>
              </EuiCallOut>
              <EuiSpacer size="m" />
            </>
          )}

          {/* Username Field */}
          <div className={styles.fieldWrapper}>
            <EuiFieldText
              placeholder={'Username'}
              value={username}
              onChange={onUsernameChange}
              onBlur={handleUsernameBlur}
              aria-label={'Username'}
              isInvalid={!!usernameError}
              disabled={isLoading || isSuccess}
              prepend={<EuiIcon type="user" />}
              fullWidth
            />
            {usernameError && (
              <EuiText size="xs" color="danger" className={styles.errorText}>
                <p>{usernameError}</p>
              </EuiText>
            )}
          </div>

          {/* Password Field */}
          <div className={styles.fieldWrapper}>
            <EuiFieldPassword
              placeholder={dict.password}
              value={password}
              onChange={onPasswordChange}
              onBlur={handlePasswordBlur}
              aria-label={dict.password}
              type={'password'}
              isInvalid={!!passwordError}
              disabled={isLoading || isSuccess}
              prepend={<EuiIcon type="lock" />}
              fullWidth
            />
            {passwordError && (
              <EuiText size="xs" color="danger" className={styles.errorText}>
                <p>{passwordError}</p>
              </EuiText>
            )}
          </div>

          {/* Submit Button */}
          <EuiButton
            color={isSuccess ? 'success' : 'primary'}
            isDisabled={!isFormValid || isLoading || isSuccess}
            fill
            onClick={handleSubmit}
            type="submit"
            fullWidth
            className={styles.submitButton}
          >
            {isLoading && (
              <>
                <EuiLoadingSpinner size="m" />
                <span style={{ marginLeft: '8px' }}>Logging in...</span>
              </>
            )}
            {isSuccess && (
              <>
                <EuiIcon type="check" />
                <span style={{ marginLeft: '8px' }}>Success!</span>
              </>
            )}
            {!isLoading && !isSuccess && 'Login'}
          </EuiButton>
        </form>
      </div>

      {/* Background decoration */}
      <div className={styles.backgroundDecoration} />
    </div>
  );
}
