"use client"

// React Dependencies
import React from "react";
// External Libraries Dependencies
import {
  EuiButton,
  EuiButtonEmpty,
  EuiFieldText,
  EuiForm,
  EuiFormRow, EuiLoadingSpinner, EuiModal,
  EuiModalBody, EuiModalFooter,
  EuiModalHeader,
  EuiModalHeaderTitle,
  EuiSelect, EuiSelectOption
} from "@elastic/eui";
// Own Dependencies
import getDict from "@/lib/dict";
import Network, { NetworkTypes } from "@/app/types/Network";
import { useToast } from "@/app/components/ToastProvider";
import { getErrorCode } from "@/lib/errorsCodes";
import { api, APIError } from "@/lib/api";
import { sanitizeInput } from "@/lib/security/validation";
const dict = getDict("de");

export const AddNetwork = (props: { closeModal: () => void}) => {
  const { addToast } = useToast();

  const [name, setName] = React.useState('');
  const [type, setType] = React.useState('');
  const [loading, setLoading] = React.useState(false);
  const [errors, setErrors] = React.useState<{ select?: string[]; text?: string[] }>({});
  const [isFormValid, setIsFormValid] = React.useState(false);

  const onNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setName(e.target.value);
  };
  const onTypeChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setType(e.target.value);
  };

  const validate = () => {
    const newErrors: { select?: string[]; text?: string[] } = {};

    if (!type) {
      newErrors.select = [dict.required];
    }

    // Network name validation
    if (name.length === 0) {
      newErrors.text = [dict.required];
    } else if (name.length < 2) {
      newErrors.text = ['Network name must be at least 2 characters'];
    } else if (name.length > 64) {
      newErrors.text = ['Network name must not exceed 64 characters'];
    } else if (!/^[a-zA-Z0-9\s\-_]+$/.test(name)) {
      newErrors.text = ['Network name can only contain letters, numbers, spaces, hyphens, and underscores'];
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  React.useEffect(() => {
    validate();
  }, [name, type]);

  React.useEffect(() => {
    setIsFormValid(Object.keys(errors).length === 0);
  }, [errors]);

  const networkTypeOptions: EuiSelectOption[] = NetworkTypes.map(type => ({
    value: type,
    text: dict[type] || type,
  }));
  const closeModal = () => {
    setLoading(false);
    setName('');
    setType('');
    setErrors({});
    props.closeModal();
  };
  async function handleSubmit() {
    setLoading(true);
    if (validate()) {
      try {
        // Sanitize input to prevent XSS
        const sanitizedName = sanitizeInput(name.trim());

        await api.getClient().post<Network>(`/edge/networks`, {
          name: sanitizedName,
          type
        });
        setLoading(false);
        addToast({
          title: "Success",
          color: "success",
          text: `${dict.networkAddToast}`,
        });
        closeModal();
      } catch (err) {
        const error = err as APIError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        setLoading(false);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkAddError}`,
        });
        closeModal();
      }
      setErrors({});
    } else {
      setLoading(false);
    }
  }

  return (
    <EuiModal
      aria-labelledby={'modalTitleId'}
      onClose={closeModal}
      initialFocus="[name=popswitch]"
    >
      <EuiModalHeader>
        <EuiModalHeaderTitle id={'modalTitleId'}>
          {dict.addNetwork}
        </EuiModalHeaderTitle>
      </EuiModalHeader>

      <EuiModalBody>
        <EuiForm component="form">
          <EuiFormRow label={dict.selectNetworkType} helpText={dict.type} error={errors.select} isInvalid={!!errors.select}>
            <EuiSelect
              options={networkTypeOptions}
              value={type}
              isInvalid={!!errors.select}
              onChange={onTypeChange}
              aria-label={dict.selectNetworkType}
              hasNoInitialSelection={true}
            />
          </EuiFormRow>
          <EuiFormRow label={dict.name} helpText={dict.name} error={errors.text} isInvalid={!!errors.text}>
            <EuiFieldText
              name={"name"}
              placeholder={dict.name}
              value={name}
              isInvalid={!!errors.text}
              onChange={(e: React.ChangeEvent<HTMLInputElement>) => onNameChange(e)}
              aria-label={dict.name}
            />
          </EuiFormRow>
        </EuiForm>
      </EuiModalBody>

      <EuiModalFooter>
        <EuiButtonEmpty onClick={closeModal}>Cancel</EuiButtonEmpty>

        <EuiButton
          type="submit"
          color={"success"}
          onClick={handleSubmit}
          form={'modalFormId'}
          isDisabled={!isFormValid || loading}
          fill
        >
          {dict.add} {loading ? <EuiLoadingSpinner size="l" /> : undefined}
        </EuiButton>
      </EuiModalFooter>
    </EuiModal>)
}
