"use client"

/**
 * Guest Network Details Component
 * Professional UI for managing guest network configurations with QR code
 *
 * Improvements:
 * - Better layout with panels and sections
 * - QR code in dedicated panel
 * - EUI buttons instead of custom Button
 * - Better spacing and visual hierarchy
 */

import getDict from "@/lib/dict";
import QRCode from "react-qr-code";
import Network, { GuestPassword } from "@/app/types/Network";
import { NetworkManagementFunctions } from "./page";
import {
  EuiConfirmModal,
  EuiDescriptionList,
  useGeneratedHtmlId,
  EuiPanel,
  EuiSpacer,
  EuiTitle,
  EuiFlexGroup,
  EuiFlexItem,
  EuiButton,
  EuiText,
  EuiBetaBadge,
  EuiStat,
  EuiIcon,
} from "@elastic/eui";
import React from "react";

export default function GuestNetworkDetails(props: { network: Network, guestPassword: GuestPassword, managementFunctions: NetworkManagementFunctions }) {
  const dict = getDict("de");

  const [isDestroyModalVisible, setIsDestroyModalVisible] = React.useState(false);
  const destroyModalTitleId = useGeneratedHtmlId();
  const closeDestroyModal = () => setIsDestroyModalVisible(false);
  const showDestroyModal = () => setIsDestroyModalVisible(true);

  return (
    <>
      {/* Guest Password and QR Code */}
      <EuiFlexGroup gutterSize="l" wrap>
        <EuiFlexItem style={{ minWidth: '300px' }}>
          <EuiPanel>
            <EuiTitle size="s">
              <h3>Guest Password</h3>
            </EuiTitle>
            <EuiSpacer size="m" />
            <EuiFlexGroup direction="column" gutterSize="s">
              <EuiFlexItem>
                <EuiText size="s" color="subdued">
                  <strong>{dict.password}:</strong>
                </EuiText>
              </EuiFlexItem>
              <EuiFlexItem>
                <EuiText size="l">
                  <strong>{props.guestPassword.password}</strong>
                </EuiText>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiPanel>
        </EuiFlexItem>

        {/* QR Code */}
        <EuiFlexItem grow={false} style={{ minWidth: '250px' }}>
          <EuiPanel>
            <EuiTitle size="s">
              <h3>WiFi QR Code</h3>
            </EuiTitle>
            <EuiSpacer size="m" />
            <EuiFlexGroup justifyContent="center">
              <EuiFlexItem grow={false}>
                <QRCode
                  value={`WIFI:S:${props.network.name};T:WPA;P:${props.guestPassword.password};H:false;`}
                  style={{maxWidth: "200px", height: "auto", width: "100%"}}
                />
              </EuiFlexItem>
            </EuiFlexGroup>
            <EuiSpacer size="s" />
            <EuiText size="s" textAlign="center" color="subdued">
              Scan to connect to {props.network.name}
            </EuiText>
          </EuiPanel>
        </EuiFlexItem>
      </EuiFlexGroup>

      <EuiSpacer size="l" />

      {/* Network Statistics */}
      <EuiPanel>
        <EuiTitle size="s">
          <h3>Network Statistics</h3>
        </EuiTitle>
        <EuiSpacer size="m" />

        <EuiFlexGroup gutterSize="l" wrap>
          {/* Client Statistics */}
          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="accent" hasShadow={false}>
              <EuiStat
                title={props.network.total_clients}
                description="Total Clients"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="users" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>

          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="success" hasShadow={false}>
              <EuiStat
                title={props.network.online_clients}
                titleColor="success"
                description="Online"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="check" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>

          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="danger" hasShadow={false}>
              <EuiStat
                title={props.network.offline_clients}
                titleColor="danger"
                description="Offline"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="cross" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>

          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="primary" hasShadow={false}>
              <EuiStat
                title={props.network.authenticated_clients}
                description="Authenticated"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="lock" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>
        </EuiFlexGroup>

        <EuiSpacer size="m" />

        {/* Traffic Statistics */}
        <EuiFlexGroup gutterSize="l" wrap>
          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="success" hasShadow={false}>
              <EuiStat
                title={(props.network.total_download / 1024).toFixed(2) + ' MB'}
                description="Total Download"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="sortDown" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>

          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="warning" hasShadow={false}>
              <EuiStat
                title={(props.network.total_upload / 1024).toFixed(2) + ' MB'}
                description="Total Upload"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="sortUp" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>

          <EuiFlexItem grow={false} style={{ minWidth: '200px' }}>
            <EuiPanel color="primary" hasShadow={false}>
              <EuiStat
                title={((props.network.total_download + props.network.total_upload) / 1024).toFixed(2) + ' MB'}
                description="Total Traffic"
                titleSize="s"
                textAlign="center"
              >
                <EuiIcon type="storage" size="xl" />
              </EuiStat>
            </EuiPanel>
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiPanel>

      <EuiSpacer size="l" />

      {/* Action Buttons */}
      <EuiPanel>
        <EuiFlexGroup gutterSize="m" wrap responsive={false}>
          <EuiFlexItem grow={false}>
            <EuiButton
              color="success"
              iconType="pencil"
              fill
              onClick={async () => {
                await props.managementFunctions.renameNetwork(props.network.id);
              }}
            >
              {dict.editName}
            </EuiButton>
          </EuiFlexItem>

          <EuiFlexItem grow={false}>
            <EuiButton
              onClick={showDestroyModal}
              color="danger"
              iconType="trash"
            >
              {dict.delete}
            </EuiButton>
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiPanel>

      {/* Delete Confirmation Modal */}
      {isDestroyModalVisible && (
        <EuiConfirmModal
          aria-labelledby={destroyModalTitleId}
          title={`${dict.remove} ${props.network.name}?`}
          titleProps={{id: destroyModalTitleId}}
          onCancel={closeDestroyModal}
          onConfirm={async () => {
            await props.managementFunctions.deleteNetwork(props.network.id);
            setIsDestroyModalVisible(false);
          }}
          cancelButtonText={dict.cancel}
          confirmButtonText={dict.confirm}
          buttonColor="danger"
          defaultFocusedButton="confirm"
        />
      )}
    </>
  );
}
