"use client"

/**
 * Network Edit Page
 * Professional UI for editing network configurations
 *
 * Improvements:
 * - Consistent EuiPage layout with header
 * - Professional breadcrumb navigation
 * - Better loading states
 * - Proper error handling
 * - Auto-refresh toggle
 */

import React, { Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
// External Libraries Dependencies
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiPageHeaderSection,
  EuiTitle,
  EuiIcon,
  EuiFlexGroup,
  EuiFlexItem,
  EuiBreadcrumbs,
  EuiLoadingSpinner,
  EuiPanel,
  EuiSpacer,
  EuiSwitch,
  EuiBetaBadge,
} from "@elastic/eui";
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import { useToast } from "@/app/components/ToastProvider";
import BusinessNetworkDetails from "./business";
import GuestNetworkDetails from "./guest";
// Other Dependencies
import getDict from "@/lib/dict";
import Network, { GuestPassword } from "@/app/types/Network";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { api } from "@/lib/api";

export type NetworkManagementFunctions = {
  renameNetwork: (id: string) => Promise<void>
  deleteNetwork: (id: string) => Promise<void>
}

function NetworkDetailsContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const id = searchParams.get("id");
  const dict = getDict("de");
  const [network, setNetwork] = React.useState<Network | null>(null);
  const [guestPassword, setGuestPassword] = React.useState<GuestPassword | null>(null);
  const [auto, setAuto] = React.useState(true);
  const { addToast } = useToast();

  const managementFunctions: NetworkManagementFunctions = {
    renameNetwork: async function (id: string): Promise<void> {
      addToast({
        title: "Warning!",
        color: "warning",
        text: `Function ${id} not implemented`,
      });
    },
    deleteNetwork: async function (id: string): Promise<void> {
      if (!id) return;

      try {
        await api.networks.deleteNetwork(id);
        addToast({
          title: "",
          color: "success",
          text: `${dict.networkDeleted}`,
        });
        router.push("/networks");
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToastFail} `,
        });
      }
    }
  }

  React.useEffect(() => {

    async function fetchNetwork() {
      if (!id) return;

      try {
        const data = await api.networks.getNetwork(id);
        if (data.type === "guest") {
          await fetchGuestPassword();
        }
        setNetwork(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `${dict.networkFetchToast}: ${id}`,
        });
        router.push("/networks");
      }
    }

    async function fetchGuestPassword() {
      if (!id) return;

      try {
        const data = await api.getClient().get<GuestPassword>(`/edge/networks/${id}/guest-password`);
        setGuestPassword(data);
      } catch (err) {
        const error = err as ApiError;
        const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
        addToast({
          title: errorCode,
          color: "danger",
          text: `Failed to fetch guest password for network with ID: ${id}`,
        });
      }
    }

    if (auto) {
      const interval = setInterval(() => {
        fetchNetwork();
      }, 5000);

      fetchNetwork();

      return () => clearInterval(interval);
    } else {
      fetchNetwork();
    }
  }, [id, router, auto]);

  // Breadcrumbs
  const breadcrumbs = [
    {
      text: dict.networks,
      href: '/networks',
      onClick: (e: React.MouseEvent) => {
        e.preventDefault();
        router.push('/networks');
      },
    },
    {
      text: network?.name || dict.editNetwork,
    },
  ];

  // Loading state
  if (!network) {
    return (
      <EuiPage paddingSize="l">
        <EuiPageBody>
          <EuiFlexGroup justifyContent="center" alignItems="center" style={{ minHeight: '200px' }}>
            <EuiFlexItem grow={false}>
              <EuiLoadingSpinner size="xl" />
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiPageBody>
      </EuiPage>
    );
  }

  return (
    <EuiPage paddingSize="l">
      <EuiPageBody>
        {/* Breadcrumbs */}
        <EuiBreadcrumbs breadcrumbs={breadcrumbs} truncate={false} />

        <EuiSpacer size="m" />

        {/* Page Header */}
        <EuiPageHeader>
          <EuiPageHeaderSection>
            <EuiFlexGroup alignItems="center" gutterSize="s">
              <EuiFlexItem grow={false}>
                <EuiIcon type="globe" size="xl" />
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiTitle size="l">
                  <h1>{network.name}</h1>
                </EuiTitle>
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiBetaBadge
                  label={dict[network.type]}
                  color={'hollow'}
                  size="m"
                />
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiPageHeaderSection>
        </EuiPageHeader>

        <EuiSpacer size="l" />

        {/* Auto-refresh Toggle */}
        <EuiPanel>
          <EuiSwitch
            label={dict.refreshData}
            checked={auto}
            onChange={() => {
              setAuto(!auto);
              addToast({
                title: auto ? dict.refreshDataDisabled : dict.refreshDataEnabled,
                color: "primary",
              });
            }}
          />
        </EuiPanel>

        <EuiSpacer size="l" />

        {/* Network Details */}
        {network.type === "guest" ? (
          guestPassword ? (
            <GuestNetworkDetails
              network={network}
              guestPassword={guestPassword}
              managementFunctions={managementFunctions}
            />
          ) : (
            <EuiPanel>
              <EuiFlexGroup justifyContent="center" alignItems="center" style={{ minHeight: '100px' }}>
                <EuiFlexItem grow={false}>
                  <EuiLoadingSpinner size="l" />
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiPanel>
          )
        ) : (
          <BusinessNetworkDetails
            network={network}
            managementFunctions={managementFunctions}
          />
        )}
      </EuiPageBody>
    </EuiPage>
  );
}

export default function NetworkDetails() {
  return (
    <AuthCheck>
      <Navigation>
        <Suspense fallback={<div>Loading...</div>}>
          <NetworkDetailsContent />
        </Suspense>
      </Navigation>
    </AuthCheck>
  );
}
