"use client"

/**
 * Networks Page
 * Professional UI showing network configurations with management
 *
 * Improvements:
 * - Consistent layout with EuiPage structure
 * - Professional header with icon
 * - Better spacing and visual hierarchy
 * - Auto-refresh toggle in panel
 * - useAuthContext instead of useAuth
 * - Better loading states
 */

import React from "react";
// Custom hooks
import { useAuthContext } from "@/features/auth";
import { useDomReady } from "@/shared/hooks";
import { useNetworks } from "@/features/networks";
// External Libraries Dependencies
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiPageHeaderSection,
  EuiTitle,
  EuiFlexItem,
  EuiButton,
  EuiFlexGroup,
  EuiLoadingSpinner,
  EuiSwitch,
  EuiSpacer,
  EuiPanel,
  EuiIcon,
} from '@elastic/eui';
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
import { useToast } from "@/app/components/ToastProvider";
import { NetworkTable } from "@/app/networks/networkTable";
import { AddNetwork } from "@/app/networks/addNetwork";
// Own Dependencies
import getDict from "@/lib/dict";

export default function Networks() {
  const dict = getDict("de");

  // State
  const [auto, setAuto] = React.useState(true);
  const [isModalVisible, setIsModalVisible] = React.useState(false);

  // Hooks
  const domReady = useDomReady();
  const { addToast } = useToast();
  const { isAuthenticated } = useAuthContext();

  // Use custom hook for network management
  const {
    networks,
    loading,
    error
  } = useNetworks({
    autoFetch: domReady && isAuthenticated,
    refetchInterval: auto ? 5000 : 0,
  });

  // Handle errors with toast
  React.useEffect(() => {
    if (error) {
      addToast({
        title: "Error",
        color: "danger",
        text: `${dict.networkFetchToastFail}`,
      });
    }
  }, [error, addToast, dict]);

  // Modal handlers
  const showModal = () => setIsModalVisible(true);
  const closeModal = () => setIsModalVisible(false);

  // Loading state
  if (!domReady) {
    return (
      <EuiPage paddingSize="none">
        <EuiPageBody>
          <EuiLoadingSpinner size="xl" />
        </EuiPageBody>
      </EuiPage>
    );
  }

  return (
    <AuthCheck>
      <Navigation>
        <EuiPage paddingSize="l">
          <EuiPageBody>
            {/* Page Header */}
            <EuiPageHeader>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="s">
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="globe" size="xl" />
                  </EuiFlexItem>
                  <EuiFlexItem grow={false}>
                    <EuiTitle size="l">
                      <h1>{dict.networks}</h1>
                    </EuiTitle>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
              <EuiPageHeaderSection>
                <EuiFlexGroup alignItems="center" gutterSize="m">
                  <EuiFlexItem grow={false}>
                    <EuiButton
                      color="success"
                      iconType="plusInCircleFilled"
                      onClick={showModal}
                      fill
                    >
                      {dict.addNetwork}
                    </EuiButton>
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiPageHeaderSection>
            </EuiPageHeader>

            <EuiSpacer size="l" />

            {/* Auto-refresh Toggle */}
            <EuiPanel>
              <EuiSwitch
                label={dict.refreshData}
                checked={auto}
                onChange={() => {
                  setAuto(!auto);
                  addToast({
                    title: auto ? dict.refreshDataDisabled : dict.refreshDataEnabled,
                    color: "primary",
                  });
                }}
              />
            </EuiPanel>

            <EuiSpacer size="l" />

            {/* Networks Table */}
            <EuiPanel>
              <NetworkTable data={networks} loading={loading} />
            </EuiPanel>

            {/* Add Network Modal */}
            {isModalVisible && <AddNetwork closeModal={closeModal} />}
          </EuiPageBody>
        </EuiPage>
      </Navigation>
    </AuthCheck>
  );
}
