"use client";

// React Dependencies
import React from "react";
// External Libraries Dependencies
import {
  EuiSpacer,
  EuiTitle,
  EuiLoadingSpinner
} from "@elastic/eui";
// Custom hooks
import { useDomReady } from "@/shared/hooks";
import { useAuthContext } from "@/features/auth";
// API Client
import { api } from "@/lib/api";
// Providers
import { useToast } from "@/app/components/ToastProvider";
// Context
import { useLoader, useLoginSplash } from "@/shared/providers";
// Components
import AuthCheck from "@/app/components/AuthCheck";
import Navigation from "@/app/components/navigation/Navigation";
// Other Dependencies
import getDict from "@/lib/dict";
import Device from "./types/Device";
import { ApiError, getErrorCode } from "@/lib/errorsCodes";
import { checkResponseShape } from "@/lib/checkResponseShape";
import PDFMapEditor from "../src/features/pdf-editor/components/PDFMapEditor";
import CardsDashboard from "../src/features/dashboard/components/Cards";

const dict = getDict("de");

export default function Dashboard() {

  const { addToast } = useToast();
  const domReady = useDomReady();
  const { setLoading, globalAps, setGlobalAps } = useLoader();
  const [devices, setDevices] = React.useState<Device[]>([]);
  const { isAuthenticated } = useAuthContext();
  const { hideLoginSplash } = useLoginSplash();
  async function fetchAPs() {
    try {
      setLoading(true);
      const data = await api.accessPoints.getAccessPoints();
      // Dont rely on backend, double check response schema
      if (checkResponseShape(data)) {
        setGlobalAps(data);
      } else {
        addToast({
          title: 'Fetch Error',
          color: "danger",
          text: dict.apFetchToastFail,
        });
      }
      setLoading(false);
    } catch (err) {
      const error = err as ApiError;
      const errorCode = getErrorCode('ACCESS_POINT_EXCEPTION', error.status);
      setLoading(false);
      addToast({
        title: errorCode,
        color: "danger",
        text: dict.apFetchToastFail,
      });
    }
  }
  // Hide login splash when dashboard mounts
  React.useEffect(() => {
    // Small delay to ensure page has rendered before hiding splash
    const timer = setTimeout(() => {
      hideLoginSplash();
    }, 100);

    return () => clearTimeout(timer);
  }, [hideLoginSplash]);

  React.useEffect(() => {
    if (!domReady || !isAuthenticated) return;

    const interval = setInterval(() => {
      fetchAPs();
    }, 5000);

    fetchAPs();
    return () => clearInterval(interval);
  }, [domReady, isAuthenticated]);
  if (!domReady) {
    return <EuiLoadingSpinner size="s"/>
  }
  return (
    <AuthCheck>
      <Navigation>
        <EuiTitle><h1>{dict.dashboard}</h1></EuiTitle>
        <EuiSpacer />
        <CardsDashboard devices={devices} dict={dict}/>
        <EuiSpacer />
        <PDFMapEditor aps={globalAps} />
      </Navigation>
    </AuthCheck>)
};
