import React from 'react';
import {
  EuiPanel,
  EuiFlexGroup,
  EuiFlexItem,
  EuiText,
  EuiIcon,
  EuiButtonIcon,
} from '@elastic/eui';
import { AccessPointMarkerPoint } from '@/features/access-points';
import AP from '@/app/types/AP';
import { AccessPointEditForm } from './AccessPointEditForm';
import getDict from '@/lib/dict';

const dict = getDict('de');

interface AccessPointItemProps {
  ap: AccessPointMarkerPoint;
  type: AP | undefined;
  isEditing: boolean;
  editingNodeName: string;
  onStartEdit: (ap: AccessPointMarkerPoint) => void;
  onSave: () => void;
  onCancel: () => void;
  onEditNameChange: (name: string) => void;
  onKeyPress: (e: React.KeyboardEvent<HTMLInputElement>) => void;
}

export const AccessPointItem: React.FC<AccessPointItemProps> = ({
  ap,
  type,
  isEditing,
  editingNodeName,
  onStartEdit,
  onSave,
  onCancel,
  onEditNameChange,
  onKeyPress,
}) => {
  return (
    <EuiPanel
      paddingSize="s"
      hasShadow={false}
      hasBorder
      style={{
        transition: 'all 0.2s ease',
        borderColor: '#d3dae6',
      }}
      onMouseEnter={(e: React.MouseEvent<HTMLDivElement>) => {
        if (!isEditing) {
          e.currentTarget.style.borderColor = '#00BFB3';
          e.currentTarget.style.boxShadow = '0 2px 6px rgba(0, 191, 179, 0.1)';
        }
      }}
      onMouseLeave={(e: React.MouseEvent<HTMLDivElement>) => {
        if (!isEditing) {
          e.currentTarget.style.borderColor = '#d3dae6';
          e.currentTarget.style.boxShadow = 'none';
        }
      }}
    >
      <EuiFlexGroup alignItems="center" gutterSize="s" responsive={false}>
        {/* Icon with Status Badge */}
        <EuiFlexItem grow={false}>
          <div style={{ position: 'relative', display: 'inline-block' }}>
            <EuiIcon type="bullseye" size="l" color="primary" />
            <div
              style={{
                position: 'absolute',
                bottom: '-2px',
                right: '-2px',
                width: '10px',
                height: '10px',
                borderRadius: '50%',
                backgroundColor: type?.online ? '#10b981' : '#ef4444',
                border: '2px solid white',
              }}
            />
          </div>
        </EuiFlexItem>

        {/* Content Area - Edit Form or Display */}
        <EuiFlexItem>
          {isEditing ? (
            <AccessPointEditForm
              value={editingNodeName}
              onChange={onEditNameChange}
              onKeyPress={onKeyPress}
              onSave={onSave}
              onCancel={onCancel}
            />
          ) : (
            <div>
              <EuiText
                size="s"
                style={{ cursor: 'pointer' }}
                onClick={(e) => {
                  e.stopPropagation();
                  onStartEdit(ap);
                }}
              >
                <strong>{ap.name}</strong>
              </EuiText>
              <EuiText size="xs" color="subdued" style={{ marginTop: '2px' }}>
                {dict.floor}: {ap.floor || 0}
              </EuiText>
            </div>
          )}
        </EuiFlexItem>

        {/* Edit Button */}
        <EuiFlexItem grow={false}>
          {!isEditing && (
            <EuiButtonIcon
              iconType="pencil"
              color="text"
              size="s"
              onClick={(e) => {
                e.stopPropagation();
                onStartEdit(ap);
              }}
              aria-label="Edit floor number"
            />
          )}
        </EuiFlexItem>
      </EuiFlexGroup>
    </EuiPanel>
  );
};