import React, { useState, useEffect, useRef } from 'react';
import {
  EuiPanel,
  EuiSpacer,
  EuiTourStep,
  EuiButton,
  EuiButtonEmpty,
  EuiFlexGroup,
  EuiFlexItem
} from '@elastic/eui';
import { AccessPointMarkerPoint } from '@/features/access-points';
import AP from '@/app/types/AP';
import { AccessPointListHeader } from './AccessPointListHeader';
import { AccessPointItem } from './AccessPointItem';

interface CurrentPageAccessPointsProps {
  currentPageAccessPoints: AccessPointMarkerPoint[];
  setAccessPoints: (updater: (prev: AccessPointMarkerPoint[]) => AccessPointMarkerPoint[]) => void;
  accessPointTypes: AP[];
  editingNodeId: string | null;
  startEditingNode: (ap: AccessPointMarkerPoint) => void;
  editingNodeName: string;
  setEditingNodeName: (name: string) => void;
  setEditingNodeId: (id: string | null) => void;
}

const TOUR_STORAGE_KEY = 'accessPointFloorTourCompleted';

export const CurrentPageAccessPoints: React.FC<CurrentPageAccessPointsProps> = React.memo(({
  currentPageAccessPoints,
  setAccessPoints,
  accessPointTypes,
  editingNodeId,
  startEditingNode,
  editingNodeName,
  setEditingNodeName,
  setEditingNodeId
}) => {
  // Tour state
  const [isTourOpen, setIsTourOpen] = useState(false);
  const [currentStep, setCurrentStep] = useState(0);
  const firstItemRef = useRef<HTMLDivElement>(null);

  // Check if tour should be shown (only on first visit and when there are access points)
  useEffect(() => {
    if (typeof window === 'undefined') return;

    const tourCompleted = localStorage.getItem(TOUR_STORAGE_KEY);
    const hasAccessPoints = currentPageAccessPoints.length > 0;

    // Show tour if not completed and there are access points
    if (!tourCompleted && hasAccessPoints) {
      // Delay to ensure DOM is ready
      const timer = setTimeout(() => {
        console.log('[CurrentPageAccessPoints Tour] Opening tour');
        setIsTourOpen(true);
      }, 500);
      return () => clearTimeout(timer);
    }
  }, [currentPageAccessPoints.length]);

  // Debug tour state
  useEffect(() => {
    console.log('[CurrentPageAccessPoints Tour] isTourOpen:', isTourOpen, 'currentStep:', currentStep);
  }, [isTourOpen, currentStep]);

  // Mark tour as completed
  const finishTour = () => {
    console.log('[CurrentPageAccessPoints Tour] Finishing tour');
    setIsTourOpen(false);
    if (typeof window !== 'undefined') {
      localStorage.setItem(TOUR_STORAGE_KEY, 'true');
    }
  };
  // Cancel editing
  const cancelEditingNode = () => {
    setEditingNodeId(null);
    setEditingNodeName('');
  };
  // Save node floor value
  const saveNodeName = () => {
    if (editingNodeName.trim()) {
      setAccessPoints((prev) =>
        prev.map((ap) =>
          ap.id === editingNodeId
            ? { ...ap, floor: editingNodeName.trim() }
            : ap
        )
      );
    }
    setEditingNodeId(null);
    setEditingNodeName('');
  };

  // Handle keyboard shortcuts for editing
  const handleEditKeyPress = (e: React.KeyboardEvent<HTMLInputElement>) => {
    e.stopPropagation();
    if (e.key === 'Enter') {
      e.preventDefault();
      saveNodeName();
    } else if (e.key === 'Escape') {
      e.preventDefault();
      cancelEditingNode();
    }
  };

  if (currentPageAccessPoints.length === 0) return null;

  // Tour navigation handlers
  const handleTourNext = () => {
    if (currentStep < 1) {
      setCurrentStep(currentStep + 1);
    } else {
      finishTour();
    }
  };

  const handleTourPrev = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  // Tour steps configuration
  const tourSteps = [
    {
      step: 1,
      title: 'Etage bearbeiten',
      content: 'Klicken Sie auf das Stift-Symbol oder den Access-Point-Namen, um die Etagennummer zu bearbeiten. Sie können verschiedene Etagennummern zuweisen, um Ihre Access Points nach Gebäudeebenen zu organisieren.',
    },
    {
      step: 2,
      title: 'Tastenkombinationen',
      content: 'Beim Bearbeiten können Sie Tastenkombinationen verwenden: Eingabe zum Speichern von Änderungen, Escape zum Abbrechen der Bearbeitung. Oder verwenden Sie die Häkchen- und Kreuz-Schaltflächen.',
    }
  ];

  // Render tour navigation controls
  const renderTourControls = (step: number) => {
    const isLastStep = step === tourSteps.length - 1;

    return (
      <EuiFlexGroup justifyContent="spaceBetween" gutterSize="s">
        <EuiFlexItem grow={false}>
          <EuiButtonEmpty
            size="s"
            onClick={handleTourPrev}
            disabled={step === 0}
            iconType="arrowLeft"
          >
            Zurück
          </EuiButtonEmpty>
        </EuiFlexItem>
        <EuiFlexItem grow={false}>
          <EuiButtonEmpty size="s" onClick={finishTour} color="text">
            Tour beenden
          </EuiButtonEmpty>
        </EuiFlexItem>
        <EuiFlexItem grow={false}>
          <EuiButton size="s" onClick={handleTourNext} fill>
            {isLastStep ? 'Fertig' : `Weiter (${step + 1}/${tourSteps.length})`}
          </EuiButton>
        </EuiFlexItem>
      </EuiFlexGroup>
    );
  };

  // Reset tour (for testing/help)
  const resetTour = () => {
    console.log('[CurrentPageAccessPoints Tour] Resetting tour');
    if (typeof window !== 'undefined') {
      localStorage.removeItem(TOUR_STORAGE_KEY);
    }
    setCurrentStep(0);
    setIsTourOpen(true);
  };

  return (
    <div style={{ padding: '24px', background: '#f5f7fa' }}>
      <EuiPanel paddingSize="m" hasShadow={false} hasBorder>
        <EuiFlexGroup alignItems="center" justifyContent="spaceBetween" gutterSize="s" responsive={false}>
          <EuiFlexItem>
            <AccessPointListHeader />
          </EuiFlexItem>
          <EuiFlexItem grow={false}>
            <EuiButtonEmpty
              size="xs"
              iconType="questionInCircle"
              onClick={resetTour}
              color="primary"
            >
              Hilfe
            </EuiButtonEmpty>
          </EuiFlexItem>
        </EuiFlexGroup>
        <EuiSpacer size="s" />
        <div style={{ display: 'flex', flexDirection: 'column', gap: '6px' }}>
          {currentPageAccessPoints.map((ap, index) => {
            const type = accessPointTypes.find((t) => t.id === ap.type);
            const isEditing = editingNodeId === ap.id;
            const isFirstItem = index === 0;

            const itemContent = (
              <AccessPointItem
                key={ap.id}
                ap={ap}
                type={type}
                isEditing={isEditing}
                editingNodeName={editingNodeName}
                onStartEdit={startEditingNode}
                onSave={saveNodeName}
                onCancel={cancelEditingNode}
                onEditNameChange={setEditingNodeName}
                onKeyPress={handleEditKeyPress}
              />
            );

            // Wrap first item with tour steps
            if (isFirstItem) {
              return (
                <div key={ap.id} ref={firstItemRef}>
                  <EuiTourStep
                    content={
                      <div>
                        <p>{tourSteps[0].content}</p>
                        <EuiSpacer size="s" />
                        {renderTourControls(0)}
                      </div>
                    }
                    isStepOpen={isTourOpen && currentStep === 0}
                    minWidth={320}
                    onFinish={finishTour}
                    step={1}
                    stepsTotal={tourSteps.length}
                    title={tourSteps[0].title}
                    anchorPosition="rightUp"
                  >
                    <EuiTourStep
                      content={
                        <div>
                          <p>{tourSteps[1].content}</p>
                          <EuiSpacer size="s" />
                          {renderTourControls(1)}
                        </div>
                      }
                      isStepOpen={isTourOpen && currentStep === 1}
                      minWidth={320}
                      onFinish={finishTour}
                      step={2}
                      stepsTotal={tourSteps.length}
                      title={tourSteps[1].title}
                      anchorPosition="rightUp"
                    >
                      {itemContent}
                    </EuiTourStep>
                  </EuiTourStep>
                </div>
              );
            }

            return itemContent;
          })}
        </div>
      </EuiPanel>
    </div>
  );
}, (prevProps, nextProps) => {
  // Only re-render if access points, editing state, or editing name changed
  return (
    prevProps.editingNodeId === nextProps.editingNodeId &&
    prevProps.editingNodeName === nextProps.editingNodeName &&
    prevProps.currentPageAccessPoints.length === nextProps.currentPageAccessPoints.length &&
    prevProps.currentPageAccessPoints.every((ap, i) =>
      ap.id === nextProps.currentPageAccessPoints[i]?.id &&
      ap.name === nextProps.currentPageAccessPoints[i]?.name &&
      ap.floor === nextProps.currentPageAccessPoints[i]?.floor
    )
  );
});

CurrentPageAccessPoints.displayName = 'CurrentPageAccessPoints';
