/**
 * Access Point Card Component
 *
 * Displays a single access point type with:
 * - Icon with status indicator
 * - Name and description
 * - Count badge (showing current/max)
 * - Lock/grab icon
 *
 * Supports drag-and-drop functionality and hover effects.
 */

import React from 'react';
import {
  EuiPanel,
  EuiFlexGroup,
  EuiFlexItem,
  EuiText,
  EuiBadge,
  EuiIcon,
} from '@elastic/eui';
import { AccessPointType } from './types';
import { COLORS } from './constants';
import getDict from '@/lib/dict';

const dict = getDict("de");

export interface AccessPointCardProps {
  type: AccessPointType;
  isAtLimit: boolean;
  currentCount: number;
  onDragStart: (e: React.DragEvent) => void;
  onTouchStart: (e: React.TouchEvent) => void;
  children: React.ReactNode; // For the icon (with or without tour)
}

export const AccessPointCard: React.FC<AccessPointCardProps> = ({
  type,
  isAtLimit,
  currentCount,
  onDragStart,
  onTouchStart,
  children,
}) => (
  <EuiPanel
    paddingSize="s"
    hasShadow={false}
    hasBorder
    style={{
      cursor: isAtLimit ? 'not-allowed' : 'grab',
      opacity: isAtLimit ? 0.6 : 1,
      backgroundColor: isAtLimit ? COLORS.background.disabled : COLORS.background.normal,
      transition: 'all 0.2s ease',
      borderWidth: '2px',
      borderStyle: 'solid',
      borderColor: type.online ? COLORS.border.online : COLORS.border.offline,
    }}
    onDragStart={onDragStart}
    onTouchStart={onTouchStart}
    onMouseEnter={(e: React.MouseEvent<HTMLDivElement>) => {
      if (!isAtLimit) {
        e.currentTarget.style.boxShadow = '0 2px 8px rgba(0, 191, 179, 0.15)';
      }
    }}
    onMouseLeave={(e: React.MouseEvent<HTMLDivElement>) => {
      if (!isAtLimit) {
        e.currentTarget.style.boxShadow = 'none';
      }
    }}
  >
    <EuiFlexGroup alignItems="center" gutterSize="s" responsive={false}>
      <EuiFlexItem grow={false}>{children}</EuiFlexItem>

      <EuiFlexItem>
        <EuiText size="s">
          <strong style={{ color: isAtLimit ? COLORS.text.subdued : COLORS.text.normal }}>
            {type.name}
          </strong>
        </EuiText>
        <EuiText size="xs" color="subdued" style={{ marginTop: '2px' }}>
          {isAtLimit ? dict.limit : dict.drag}
        </EuiText>
      </EuiFlexItem>

      <EuiFlexItem grow={false}>
        <EuiBadge
          color={isAtLimit ? 'danger' : currentCount > 0 ? 'success' : 'default'}
          style={{ minWidth: '42px', textAlign: 'center' }}
        >
          {currentCount}/1
        </EuiBadge>
      </EuiFlexItem>

      <EuiFlexItem grow={false} style={{ minWidth: '24px' }}>
        <EuiIcon
          type={isAtLimit ? 'lock' : 'grab'}
          color="subdued"
          size="m"
        />
      </EuiFlexItem>
    </EuiFlexGroup>
  </EuiPanel>
);