/**
 * AccessPointTypes Component - Main Module
 *
 * Displays available access point types with drag-and-drop functionality
 * and an interactive tour for first-time users.
 *
 */

import React from 'react';
import {
  EuiPanel,
  EuiTitle,
  EuiSpacer,
  EuiFlexGroup,
  EuiFlexItem,
  EuiIcon,
  EuiToolTip,
  EuiTourStep,
  EuiButtonEmpty,
} from '@elastic/eui';
import getDict from '@/lib/dict';

// Internal imports
import { AccessPointTypesProps, AccessPointType } from './types';
import { COLORS, TOUR_STEPS } from './constants';
import { useTourManager } from './useTourManager';
import { TourNavigationControls } from './TourNavigationControls';
import { AccessPointIcon } from './AccessPointIcon';
import { AccessPointCard } from './AccessPointCard';

const dict = getDict("de");

/**
 * AccessPointTypes Component
 *
 * Main component that renders the list of available access point types
 * with an interactive tour system.
 */
export const AccessPointTypes = React.memo<AccessPointTypesProps>(
  ({
    accessPointTypes,
    isTypeAtLimit,
    getTypeCount,
    handleDragStart,
    handleTouchStartType,
    accessPoints,
  }) => {
    const {
      isTourActive,
      currentTourStep,
      handleTourNext,
      handleTourPrev,
      handleTourEnd,
      resetTour,
    } = useTourManager(accessPointTypes.length);

    /**
     * Renders the icon with optional tour wrapping for steps 2 and 3
     */
    const renderIcon = (
      type: AccessPointType,
      isAtLimit: boolean,
      isFirstItem: boolean
    ) => {
      const icon = (
        <AccessPointIcon
          isAtLimit={isAtLimit}
          isOnline={type.online}
        />
      );

      // Step 2: Drag & Drop tour
      if (isTourActive && currentTourStep === 1 && isFirstItem) {
        return (
          <EuiTourStep
            content={
              <div>
                <p>{TOUR_STEPS[1].content}</p>
                <EuiSpacer size="s" />
                <TourNavigationControls
                  currentStep={currentTourStep}
                  totalSteps={TOUR_STEPS.length}
                  onPrev={handleTourPrev}
                  onEnd={handleTourEnd}
                  onNext={handleTourNext}
                />
              </div>
            }
            isStepOpen={true}
            minWidth={320}
            onFinish={handleTourEnd}
            step={2}
            stepsTotal={TOUR_STEPS.length}
            title={TOUR_STEPS[1].title}
            anchorPosition="rightUp"
          >
            {icon}
          </EuiTourStep>
        );
      }

      // Step 3: Status indicator tour
      if (isTourActive && currentTourStep === 2 && isFirstItem) {
        return (
          <EuiTourStep
            content={
              <div>
                <p>{TOUR_STEPS[2].content}</p>
                <EuiSpacer size="s" />
                <TourNavigationControls
                  currentStep={currentTourStep}
                  totalSteps={TOUR_STEPS.length}
                  onPrev={handleTourPrev}
                  onEnd={handleTourEnd}
                  onNext={handleTourNext}
                />
              </div>
            }
            isStepOpen={true}
            minWidth={320}
            onFinish={handleTourEnd}
            step={3}
            stepsTotal={TOUR_STEPS.length}
            title={TOUR_STEPS[2].title}
            anchorPosition="rightUp"
          >
            {icon}
          </EuiTourStep>
        );
      }

      return icon;
    };

    /**
     * Renders the complete card with optional tour wrapping for step 4
     */
    const renderCard = (
      type: AccessPointType,
      isAtLimit: boolean,
      currentCount: number,
      isFirstItem: boolean
    ) => {
      const icon = renderIcon(type, isAtLimit, isFirstItem);

      const card = (
        <AccessPointCard
          type={type}
          isAtLimit={isAtLimit}
          currentCount={currentCount}
          onDragStart={(e) => handleDragStart(e, type.id)}
          onTouchStart={(e) => handleTouchStartType(e, type.id)}
        >
          {icon}
        </AccessPointCard>
      );

      // Step 4: Limit & Badge tour (show on locked or first item)
      const showLimitTour = isTourActive && currentTourStep === 3 && (isAtLimit || isFirstItem);

      if (showLimitTour) {
        return (
          <EuiTourStep
            content={
              <div>
                <p>{TOUR_STEPS[3].content}</p>
                <EuiSpacer size="s" />
                <TourNavigationControls
                  currentStep={currentTourStep}
                  totalSteps={TOUR_STEPS.length}
                  onPrev={handleTourPrev}
                  onEnd={handleTourEnd}
                  onNext={handleTourNext}
                />
              </div>
            }
            isStepOpen={true}
            minWidth={320}
            onFinish={handleTourEnd}
            step={4}
            stepsTotal={TOUR_STEPS.length}
            title={TOUR_STEPS[3].title}
            anchorPosition="leftUp"
          >
            <EuiToolTip content={isAtLimit ? dict.limit : dict.drag} position="left">
              {card}
            </EuiToolTip>
          </EuiTourStep>
        );
      }

      return (
        <EuiToolTip content={isAtLimit ? dict.limit : dict.drag} position="left">
          {card}
        </EuiToolTip>
      );
    };

    return (
      <div style={{ background: COLORS.background.container, padding: '24px' }}>
        <EuiPanel paddingSize="m" hasShadow={false} hasBorder>
          {/* Header with Tour Step 1 */}
          <EuiFlexGroup
            alignItems="center"
            justifyContent="spaceBetween"
            gutterSize="s"
            responsive={false}
          >
            <EuiFlexItem grow={false}>
              <EuiFlexGroup alignItems="center" gutterSize="s" responsive={false}>
                <EuiFlexItem grow={false}>
                  <EuiTourStep
                    content={
                      <div>
                        <p>{TOUR_STEPS[0].content}</p>
                        <EuiSpacer size="s" />
                        <TourNavigationControls
                          currentStep={currentTourStep}
                          totalSteps={TOUR_STEPS.length}
                          onPrev={handleTourPrev}
                          onEnd={handleTourEnd}
                          onNext={handleTourNext}
                          showBack={false}
                        />
                      </div>
                    }
                    isStepOpen={isTourActive && currentTourStep === 0}
                    minWidth={300}
                    onFinish={handleTourEnd}
                    step={1}
                    stepsTotal={TOUR_STEPS.length}
                    title={TOUR_STEPS[0].title}
                    anchorPosition="rightUp"
                  >
                    <EuiIcon type="bullseye" size="m" color="primary" />
                  </EuiTourStep>
                </EuiFlexItem>
                <EuiFlexItem>
                  <EuiTitle size="xs">
                    <h3>{dict.available + ' ' + dict.accessPoint}</h3>
                  </EuiTitle>
                </EuiFlexItem>
              </EuiFlexGroup>
            </EuiFlexItem>

            {/* Help button to restart tour */}
            <EuiFlexItem grow={false}>
              <EuiButtonEmpty
                size="xs"
                iconType="questionInCircle"
                onClick={resetTour}
                color="primary"
              >
                Hilfe
              </EuiButtonEmpty>
            </EuiFlexItem>
          </EuiFlexGroup>

          <EuiSpacer size="s" />

          {/* Access Point Types List */}
          <EuiFlexGroup direction="column" gutterSize="s">
            {accessPointTypes.map((type, index) => {
              const isAtLimit = isTypeAtLimit(type.id, accessPoints);
              const currentCount = getTypeCount(type.id, accessPoints);
              const isFirstItem = index === 0;

              return (
                <EuiFlexItem key={type.id}>
                  {renderCard(type, isAtLimit, currentCount, isFirstItem)}
                </EuiFlexItem>
              );
            })}
          </EuiFlexGroup>
        </EuiPanel>
      </div>
    );
  },
  (prevProps, nextProps) => {
    // Only re-render if access points changed (count or types)
    return (
      prevProps.accessPoints.length === nextProps.accessPoints.length &&
      prevProps.accessPointTypes.length === nextProps.accessPointTypes.length &&
      prevProps.accessPoints.every(
        (ap, i) =>
          ap.id === nextProps.accessPoints[i]?.id &&
          ap.type === nextProps.accessPoints[i]?.type
      )
    );
  }
);

AccessPointTypes.displayName = 'AccessPointTypes';
