/**
 * Custom hook for managing tour state and behavior
 */

import { useState, useEffect, useCallback } from 'react';
import { UseTourManagerReturn } from './types';
import { TOUR_LOCAL_STORAGE_KEY, TOUR_AUTO_START_DELAY, TOUR_STEPS } from './constants';

/**
 * useTourManager Hook
 *
 * Manages the interactive tour state including:
 * - Auto-starting the tour for first-time visitors
 * - Navigation between tour steps
 * - Persisting completion status to localStorage
 *
 * @param accessPointTypesCount - Number of access point types available
 * @returns Tour state and control functions
 */
export function useTourManager(accessPointTypesCount: number): UseTourManagerReturn {
  const [isTourActive, setIsTourActive] = useState(false);
  const [currentTourStep, setCurrentTourStep] = useState(0);

  // Auto-start tour on first visit
  useEffect(() => {
    const tourCompleted = localStorage.getItem(TOUR_LOCAL_STORAGE_KEY);
    if (!tourCompleted && accessPointTypesCount > 0) {
      const timer = setTimeout(() => {
        setIsTourActive(true);
      }, TOUR_AUTO_START_DELAY);
      return () => clearTimeout(timer);
    }
  }, [accessPointTypesCount]);

  const handleTourNext = useCallback(() => {
    if (currentTourStep < TOUR_STEPS.length - 1) {
      setCurrentTourStep(prev => prev + 1);
    } else {
      handleTourEnd();
    }
  }, [currentTourStep]);

  const handleTourPrev = useCallback(() => {
    if (currentTourStep > 0) {
      setCurrentTourStep(prev => prev - 1);
    }
  }, [currentTourStep]);

  const handleTourEnd = useCallback(() => {
    setIsTourActive(false);
    setCurrentTourStep(0);
    localStorage.setItem(TOUR_LOCAL_STORAGE_KEY, 'true');
  }, []);

  const resetTour = useCallback(() => {
    setIsTourActive(true);
    setCurrentTourStep(0);
  }, []);

  return {
    isTourActive,
    currentTourStep,
    handleTourNext,
    handleTourPrev,
    handleTourEnd,
    resetTour,
  };
}