import AP from '@/app/types/AP';
import { useLoader } from '@/shared/providers';
import { useState } from 'react';

interface AccessPoint {
  id: string;
  type: string;
  name: string;
  [key: string]: any;
}

interface DragItem {
  type: string;
  [key: string]: any;
}

export const useAccessPoints = (aps: AP[]) => {
  const [accessPoints, setAccessPoints] = useState<AccessPoint[]>([]);
  const [draggedItem, setDraggedItem] = useState<DragItem | null>(null);
  const [allowNodeMovement, setAllowNodeMovement] = useState(true);
  const [editingNodeId, setEditingNodeId] = useState<string | null>(null);
  const [editingNodeName, setEditingNodeName] = useState('');
  const [contextMenuNodeId, setContextMenuNodeId] = useState<string | null>(null);
  const { globalAps } = useLoader();
  const uniqueIds = [...new Set(aps.map(ap => ap.type))];
  const accessPointLimits = {
    total: aps.length,
    perType: Object.fromEntries(uniqueIds.map((type: string) => [type, 1])),
  };
  const accessPointTypes = aps;
  const canAddAccessPoint = (type: string) => {
    const totalCount = accessPoints.length;
    const typeCount = accessPoints.filter(ap => ap.type === type).length;
    const perTypeLimit = accessPointLimits.perType[type] ?? 1;
    return {
      canAdd: totalCount < accessPointLimits.total && typeCount < perTypeLimit,
      limits: {
        total: { current: totalCount, max: accessPointLimits.total },
        type: { current: typeCount, max: perTypeLimit },
      },
    };
  };
  const getLimitStatus = (type: string) => {
    const { canAdd, limits } = canAddAccessPoint(type);
    return {
      canAdd,
      totalProgress: (limits.total.current / limits.total.max) * 100,
      typeProgress: (limits.type.current / limits.type.max) * 100,
      messages: {
        total: `${limits.total.current}/${limits.total.max}`,
        type: `${limits.type.current}/${limits.type.max} ${type}`,
      }
    };
  };

  const generateDeviceDetails = async (id: string) => {
    await globalAps;
    // Find the current access point with updated status
    const currentAccessPoint = globalAps?.find(ap => id === ap.id);
    const isOnline = currentAccessPoint?.online ?? false;
    return {
      online: isOnline
    }
  };

  const removeAccessPoint = (id: string) => {
    setAccessPoints(accessPoints.filter(ap => ap.id !== id));
  };

  const startEditingNode = (accessPoint: AccessPoint) => {
    setEditingNodeId(accessPoint.id);
    setEditingNodeName(accessPoint.name);
  };

  const saveNodeName = () => {
    if (editingNodeName.trim()) {
      setAccessPoints(prev => prev.map(ap =>
        ap.id === editingNodeId
          ? { ...ap, floor: editingNodeName.trim() }
          : ap
      ));
    }
    setEditingNodeId(null);
    setEditingNodeName('');
  };

  const cancelEditingNode = () => {
    setEditingNodeId(null);
    setEditingNodeName('');
  };

  return {
    accessPoints,
    setAccessPoints,
    accessPointTypes,
    draggedItem,
    setDraggedItem,
    allowNodeMovement,
    setAllowNodeMovement,
    editingNodeId,
    editingNodeName,
    setEditingNodeName,
    contextMenuNodeId,
    setContextMenuNodeId,
    generateDeviceDetails,
    removeAccessPoint,
    startEditingNode,
    saveNodeName,
    cancelEditingNode,
    canAddAccessPoint,
    accessPointLimits,
    getLimitStatus,
  };
};
