import { useState, useEffect, useCallback } from 'react';
import { api, APIError } from '@/lib/api';
import type { AccessPoint } from '@/lib/api/endpoints/accessPoints';

/**
 * Hook state for managing access points
 */
interface UseAccessPointsState {
  accessPoints: AccessPoint[];
  loading: boolean;
  error: APIError | null;
}

/**
 * Options for the useAccessPointsAPI hook
 */
interface UseAccessPointsOptions {
  /** Auto-fetch access points on mount */
  autoFetch?: boolean;
  /** Refetch interval in milliseconds (0 = no polling) */
  refetchInterval?: number;
  /** Filter by network ID */
  networkId?: string;
  /** Filter by type */
  type?: string;
}

/**
 * useAccessPointsAPI Hook
 *
 * A convenient hook for managing access points with common patterns:
 * - Fetching access points list
 * - Creating new access points
 * - Updating access points
 * - Deleting access points
 * - Auto-refresh/polling
 * - Loading and error states
 * - Filtering by network or type
 *
 * @param options - Configuration options
 * @returns Access point management utilities
 *
 */
export const useAccessPointsAPI = (options: UseAccessPointsOptions = {}) => {
  const { autoFetch = true, refetchInterval = 0, networkId, type } = options;

  const [state, setState] = useState<UseAccessPointsState>({
    accessPoints: [],
    loading: false,
    error: null,
  });

  /**
   * Fetch all access points from the API
   */
  const fetchAccessPoints = useCallback(async () => {
    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      let accessPoints: AccessPoint[];

      // Apply filters if provided
      if (networkId) {
        accessPoints = await api.accessPoints.getAccessPointsByNetwork(networkId);
      } else if (type) {
        accessPoints = await api.accessPoints.getAccessPointsByType(type);
      } else {
        accessPoints = await api.accessPoints.getAccessPoints();
      }

      setState({
        accessPoints: accessPoints || [],
        loading: false,
        error: null,
      });
    } catch (err) {
      const error = err instanceof APIError ? err : new APIError('Failed to fetch access points');
      setState(prev => ({
        ...prev,
        loading: false,
        error,
      }));
    }
  }, [networkId, type]);

  /**
   * Fetch a single access point by ID
   */
  const getAccessPoint = useCallback(async (id: string): Promise<AccessPoint | null> => {
    try {
      const accessPoint = await api.accessPoints.getAccessPoint(id);
      return accessPoint || null;
    } catch (err) {
      console.error('Failed to fetch access point:', err);
      return null;
    }
  }, []);

  /**
   * Create a new access point
   */
  const createAccessPoint = useCallback(async (accessPointData: {
    name: string;
    type?: string;
    network?: string;
    location?: { x: number; y: number; floor?: string };
    [key: string]: any;
  }): Promise<AccessPoint | null> => {
    try {
      const accessPoint = await api.accessPoints.createAccessPoint(accessPointData);

      // Optimistically add to local state
      if (accessPoint) {
        setState(prev => ({
          ...prev,
          accessPoints: [...prev.accessPoints, accessPoint],
        }));
      }

      return accessPoint || null;
    } catch (err) {
      console.error('Failed to create access point:', err);
      throw err;
    }
  }, []);

  /**
   * Update an existing access point
   */
  const updateAccessPoint = useCallback(async (
    id: string,
    data: Partial<AccessPoint>
  ): Promise<AccessPoint | null> => {
    try {
      const updated = await api.accessPoints.updateAccessPoint(id, data);

      // Optimistically update local state
      if (updated) {
        setState(prev => ({
          ...prev,
          accessPoints: prev.accessPoints.map(ap =>
            ap.id === id ? updated : ap
          ),
        }));
      }

      return updated || null;
    } catch (err) {
      console.error('Failed to update access point:', err);
      throw err;
    }
  }, []);

  /**
   * Delete an access point
   */
  const deleteAccessPoint = useCallback(async (id: string): Promise<boolean> => {
    try {
      await api.accessPoints.deleteAccessPoint(id);

      // Optimistically remove from local state
      setState(prev => ({
        ...prev,
        accessPoints: prev.accessPoints.filter(ap => ap.id !== id),
      }));

      return true;
    } catch (err) {
      console.error('Failed to delete access point:', err);
      throw err;
    }
  }, []);

  /**
   * Get access point statistics
   */
  const getAccessPointStats = useCallback(async (id: string) => {
    try {
      return await api.accessPoints.getAccessPointStats(id);
    } catch (err) {
      console.error('Failed to fetch access point stats:', err);
      return null;
    }
  }, []);

  /**
   * Refetch access points (alias for fetchAccessPoints)
   */
  const refetch = fetchAccessPoints;

  /**
   * Clear error state
   */
  const clearError = useCallback(() => {
    setState(prev => ({ ...prev, error: null }));
  }, []);

  // Auto-fetch on mount if enabled
  useEffect(() => {
    if (autoFetch) {
      fetchAccessPoints();
    }
  }, [autoFetch, fetchAccessPoints]);

  // Setup polling if interval is specified
  useEffect(() => {
    if (refetchInterval > 0) {
      const intervalId = setInterval(fetchAccessPoints, refetchInterval);
      return () => clearInterval(intervalId);
    }
  }, [refetchInterval, fetchAccessPoints]);

  return {
    // State
    accessPoints: state.accessPoints,
    loading: state.loading,
    error: state.error,

    // Actions
    fetchAccessPoints,
    refetch,
    getAccessPoint,
    createAccessPoint,
    updateAccessPoint,
    deleteAccessPoint,
    getAccessPointStats,
    clearError,
  };
};

/**
 * useAccessPointAPI Hook (for single access point)
 *
 * Simplified hook for managing a single access point
 *
 * @param accessPointId - The ID of the access point to fetch
 * @param options - Configuration options
 * @returns Single access point management utilities
 *
 */
export const useAccessPointAPI = (
  accessPointId: string | null,
  options: { autoFetch?: boolean; refetchInterval?: number } = {}
) => {
  const { autoFetch = true, refetchInterval = 0 } = options;

  const [state, setState] = useState<{
    accessPoint: AccessPoint | null;
    loading: boolean;
    error: APIError | null;
  }>({
    accessPoint: null,
    loading: false,
    error: null,
  });

  /**
   * Fetch the access point
   */
  const fetchAccessPoint = useCallback(async () => {
    if (!accessPointId) return;

    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      const accessPoint = await api.accessPoints.getAccessPoint(accessPointId);
      setState({
        accessPoint: accessPoint || null,
        loading: false,
        error: null,
      });
    } catch (err) {
      const error = err instanceof APIError ? err : new APIError('Failed to fetch access point');
      setState(prev => ({
        ...prev,
        loading: false,
        error,
      }));
    }
  }, [accessPointId]);

  /**
   * Update the access point
   */
  const updateAccessPoint = useCallback(async (data: Partial<AccessPoint>): Promise<AccessPoint | null> => {
    if (!accessPointId) return null;

    try {
      const updated = await api.accessPoints.updateAccessPoint(accessPointId, data);
      setState(prev => ({
        ...prev,
        accessPoint: updated || prev.accessPoint,
      }));
      return updated || null;
    } catch (err) {
      console.error('Failed to update access point:', err);
      throw err;
    }
  }, [accessPointId]);

  /**
   * Delete the access point
   */
  const deleteAccessPoint = useCallback(async (): Promise<boolean> => {
    if (!accessPointId) return false;

    try {
      await api.accessPoints.deleteAccessPoint(accessPointId);
      setState({
        accessPoint: null,
        loading: false,
        error: null,
      });
      return true;
    } catch (err) {
      console.error('Failed to delete access point:', err);
      throw err;
    }
  }, [accessPointId]);

  /**
   * Get access point statistics
   */
  const getStats = useCallback(async () => {
    if (!accessPointId) return null;

    try {
      return await api.accessPoints.getAccessPointStats(accessPointId);
    } catch (err) {
      console.error('Failed to fetch access point stats:', err);
      return null;
    }
  }, [accessPointId]);

  // Auto-fetch on mount or when accessPointId changes
  useEffect(() => {
    if (autoFetch && accessPointId) {
      fetchAccessPoint();
    }
  }, [autoFetch, accessPointId, fetchAccessPoint]);

  // Setup polling if interval is specified
  useEffect(() => {
    if (refetchInterval > 0 && accessPointId) {
      const intervalId = setInterval(fetchAccessPoint, refetchInterval);
      return () => clearInterval(intervalId);
    }
  }, [refetchInterval, accessPointId, fetchAccessPoint]);

  return {
    // State
    accessPoint: state.accessPoint,
    loading: state.loading,
    error: state.error,

    // Actions
    refetch: fetchAccessPoint,
    update: updateAccessPoint,
    remove: deleteAccessPoint,
    getStats,
  };
};
