"use client"
// React Dependencies
import React from 'react';
// NextJs Dependencies
import { useRouter } from "next/navigation";
import Image from "next/image";
// External Libraries Dependencies
import {
  EuiButton,
  EuiFieldText,
  EuiLoadingSpinner,
  EuiFieldPassword,
  EuiText,
  EuiTitle,
  EuiCheckbox,
  EuiSpacer,
  EuiCallOut,
  EuiIcon
} from "@elastic/eui";
// Own Dependencies
import styles from "./page.module.css";
import getDict from "@/lib/dict";
import branding from "@/lib/branding";
import { useToast } from "@/app/components/ToastProvider";
import { validateEmail } from "@/lib/security/validation";
import Cookies from "js-cookie";
import { api } from "@/lib/api";
import { AuthenticationError, RateLimitError, NetworkError } from "@/lib/api/errors";
import {SplashScreen} from "@/shared/components";
type LoginState = 'idle' | 'loading' | 'success' | 'error';

export default function Login() {
  const dict = getDict("de");
  const router = useRouter();
  const { addToast } = useToast();

  // Form state
  const [email, setEmail] = React.useState('');
  const [password, setPassword] = React.useState('');
  const [rememberMe, setRememberMe] = React.useState(false);

  // UI state
  const [loginState, setLoginState] = React.useState<LoginState>('idle');
  const [errorMessage, setErrorMessage] = React.useState('');
  const [emailError, setEmailError] = React.useState('');
  const [passwordError, setPasswordError] = React.useState('');
  const [showPassword, setShowPassword] = React.useState(false);
  const [showSplash, setShowSplash] = React.useState(false);
  const [splashFadeOut, setSplashFadeOut] = React.useState(false);

  // Validate email on blur
  const handleEmailBlur = () => {
    if (email.length === 0) return;

    const validation = validateEmail(email);
    if (!validation.valid) {
      setEmailError(validation.error || 'Invalid email');
    } else {
      setEmailError('');
    }
  };

  // Validate password on blur
  const handlePasswordBlur = () => {
    if (password.length === 0) return;

    if (password.length < 6) {
      setPasswordError('Password must be at least 6 characters');
    } else {
      setPasswordError('');
    }
  };

  // Handle form submission
  async function handleSubmit(event?: React.FormEvent) {
    event?.preventDefault();

    // Clear previous errors
    setErrorMessage('');
    setEmailError('');
    setPasswordError('');

    // Validate email
    const emailValidation = validateEmail(email);
    if (!emailValidation.valid) {
      setEmailError(emailValidation.error || 'Invalid email');
      return;
    }

    // Validate password
    if (password.length < 6) {
      setPasswordError('Password must be at least 6 characters');
      return;
    }

    setLoginState('loading');

    try {
      // Use new API client for login
      const response = await api.auth.login({
        email,
        password,
        rememberMe,
      });

      // Check for successful login
      if (response && response.apiKey) {
        // Store token in cookie (handled by interceptor, but keep for legacy compatibility)
        Cookies.set("apiKey", response.apiKey);

        // Store email for display purposes
        if (email) {
          Cookies.set("userEmail", email);
        }

        setLoginState('success');

        // Show the splash screen immediately
        setShowSplash(true);

        // Start fade out after 1.5 seconds
        setTimeout(() => {
          setSplashFadeOut(true);
        }, 1500);

        // Redirect after splash animation completes
        setTimeout(() => {
          router.push("/");
        }, 2200);
      } else {
        // Handle unsuccessful login response
        setLoginState('error');
        const errorMsg = response.error || 'Invalid credentials. Please check your email and password.';
        setErrorMessage(errorMsg);

        addToast({
          title: 'Login Failed',
          color: "danger",
          text: errorMsg,
        });
      }
    } catch (error) {
      setLoginState('error');

      // Handle specific error types
      if (error instanceof AuthenticationError) {
        const errorMsg = 'Invalid credentials. Please check your email and password.';
        setErrorMessage(errorMsg);

        addToast({
          title: 'Login Failed',
          color: "danger",
          text: errorMsg,
        });
      } else if (error instanceof RateLimitError) {
        const errorMsg = 'Too many login attempts. Please try again later.';
        setErrorMessage(errorMsg);

        addToast({
          title: 'Rate Limit Exceeded',
          color: "danger",
          text: errorMsg,
        });
      } else if (error instanceof NetworkError) {
        const errorMsg = 'Unable to connect to server. Please check your connection.';
        setErrorMessage(errorMsg);

        addToast({
          title: 'Connection Error',
          color: "danger",
          text: errorMsg,
        });
      } else {
        // Generic error
        const errorMsg = error instanceof Error ? error.message : 'An unexpected error occurred.';
        setErrorMessage(errorMsg);

        addToast({
          title: 'Error',
          color: "danger",
          text: errorMsg,
        });
      }
    }
  }

  // Handle Enter key
  React.useEffect(() => {
    const listener = (event: KeyboardEvent) => {
      if (event.code === 'Enter' || event.code === 'NumpadEnter') {
        if (password.length >= 6 && email.length >= 4 && loginState === 'idle') {
          handleSubmit();
        }
      }
    };

    document.addEventListener('keydown', listener);
    return () => {
      document.removeEventListener('keydown', listener);
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [password, email, loginState]);

  const onEmailChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setEmail(e.target.value);
    if (emailError) setEmailError('');
  };

  const onPasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setPassword(e.target.value);
    if (passwordError) setPasswordError('');
  };

  const isFormValid = email.length >= 4 && password.length >= 6 && !emailError && !passwordError;
  const isLoading = loginState === 'loading';
  const isSuccess = loginState === 'success';
  return (
    <>
      {/* Splash Screen */}
      <SplashScreen
        show={showSplash}
        fadeOut={splashFadeOut}
        title={branding.productName}
        subtitle="Welcome back!"
        showSpinner={true}
      />

      {/* Hide login form when splash screen is showing */}
      {!showSplash && (
        <div className={styles.container}>
          <div className={styles.loginCard}>
            <form className={styles.form} onSubmit={handleSubmit}>
              {/* Branding */}
              <div className={styles.branding}>
                <div className={styles.logoWrapper}>
                  <Image
                    src="/logo.svg"
                    alt="Logo"
                    width={60}
                    height={60}
                    priority
                    className={styles.logo}
                  />
                </div>
                <div className={styles.brandingText}>
                  <EuiTitle size="l">
                    <h1>{branding.productName}</h1>
                  </EuiTitle>
                  <EuiText size="s" color="subdued">
                    <p>{branding.companyName}</p>
                  </EuiText>
                </div>
              </div>

              <EuiSpacer size="l" />

              {/* Title */}
              <div className={styles.titleSection}>
                <EuiTitle size="m">
                  <h2>{dict.login}</h2>
                </EuiTitle>
                <EuiText size="s" color="subdued">
                  <p>Enter your credentials to access your account</p>
                </EuiText>
              </div>

              <EuiSpacer size="m" />

              {/* Error Message */}
              {loginState === 'error' && errorMessage && (
                <>
                  <EuiCallOut
                    title="Login Failed"
                    color="danger"
                    iconType="alert"
                    size="s"
                  >
                    <p>{errorMessage}</p>
                  </EuiCallOut>
                  <EuiSpacer size="m" />
                </>
              )}

              {/* Email Field */}
              <div className={styles.fieldWrapper}>
                <EuiFieldText
                  placeholder={dict.email}
                  value={email}
                  onChange={onEmailChange}
                  onBlur={handleEmailBlur}
                  aria-label={dict.email}
                  isInvalid={!!emailError}
                  disabled={isLoading || isSuccess}
                  prepend={<EuiIcon type="email" />}
                  fullWidth
                />
                {emailError && (
                  <EuiText size="xs" color="danger" className={styles.errorText}>
                    <p>{emailError}</p>
                  </EuiText>
                )}
              </div>

              {/* Password Field */}
              <div className={styles.fieldWrapper}>
                <EuiFieldPassword
                  placeholder={dict.password}
                  value={password}
                  onChange={onPasswordChange}
                  onBlur={handlePasswordBlur}
                  aria-label={dict.password}
                  type={showPassword ? 'dual' : 'password'}
                  isInvalid={!!passwordError}
                  disabled={isLoading || isSuccess}
                  prepend={<EuiIcon type="lock" />}
                  fullWidth
                />
                {passwordError && (
                  <EuiText size="xs" color="danger" className={styles.errorText}>
                    <p>{passwordError}</p>
                  </EuiText>
                )}
              </div>

              {/* Remember Me */}
              <div className={styles.optionsRow}>
                <EuiCheckbox
                  id="rememberMe"
                  label="Remember me"
                  checked={rememberMe}
                  onChange={(e) => setRememberMe(e.target.checked)}
                  disabled={isLoading || isSuccess}
                />
              </div>

              {/* Submit Button */}
              <EuiButton
                color={isSuccess ? 'success' : 'primary'}
                isDisabled={!isFormValid || isLoading || isSuccess}
                fill
                onClick={handleSubmit}
                type="submit"
                fullWidth
                className={styles.submitButton}
              >
                {isLoading && (
                  <>
                    <EuiLoadingSpinner size="m" />
                    <span style={{ marginLeft: '8px' }}>Logging in...</span>
                  </>
                )}
                {isSuccess && (
                  <>
                    <EuiIcon type="check" />
                    <span style={{ marginLeft: '8px' }}>Success!</span>
                  </>
                )}
                {!isLoading && !isSuccess && 'Login'}
              </EuiButton>
            </form>
          </div>

          {/* Background decoration */}
          <div className={styles.backgroundDecoration} />
        </div>
      )}
    </>
  );
}
