"use client"

import React from 'react';
import { useRouter } from "next/navigation";
import { getToken } from "@/features/auth/utils";
import { hasValidSession, logout as secureLogout } from "@/lib/security/auth";
import Cookies from "js-cookie";

interface AuthContextType {
  isAuthenticated: boolean;
  login: (username: string, apiKey: string) => void;
  logout: () => Promise<void>;
  isLoggingOut: boolean;
  showLogoutSplash: boolean;
  logoutSplashFadeOut: boolean;
  userEmail: string | null;
  hideLogoutSplash: () => void;
}

const AuthContext = React.createContext<AuthContextType | undefined>(undefined);

/**
 * AuthProvider Component
 *
 * Provides authentication state and methods to all child components
 */
export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [isAuthenticated, setIsAuthenticated] = React.useState(false);
  const [isLoggingOut, setIsLoggingOut] = React.useState(false);
  const [showLogoutSplash, setShowLogoutSplash] = React.useState(false);
  const [logoutSplashFadeOut, setLogoutSplashFadeOut] = React.useState(false);
  const [userEmail, setUserEmail] = React.useState<string | null>(null);
  const router = useRouter();

  const login = (username: string, apiKey: string) => {
    setIsAuthenticated(true);
    setUserEmail(username);
    // Store cookies
    Cookies.set("apiKey", apiKey);
    Cookies.set("username", username);
  };

  const logout = async () => {
    if (isLoggingOut) return;

    setIsLoggingOut(true);

    try {
      // Show splash screen immediately
      setShowLogoutSplash(true);

      // Call secure logout (clears HttpOnly cookies on backend)
      const success = await secureLogout();

      // Clear any legacy cookies regardless of success
      Cookies.remove("apiKey");
      Cookies.remove("username");

      // Reset state
      setIsAuthenticated(false);
      setUserEmail(null);

      // Keep splash visible for at least 2 seconds before navigating
      setTimeout(() => {
        router.push("/login");
        // Start fade out after navigation begins
        setTimeout(() => {
          setLogoutSplashFadeOut(true);
        }, 1000);
        // Splash will be hidden by the login page when it mounts
      }, 2000);

    } catch (error) {
      console.error('[AUTH] Logout error:', error);

      // Even on error, clear client state and show splash
      Cookies.remove("apiKey");
      Cookies.remove("username");
      setIsAuthenticated(false);
      setUserEmail(null);

      // Keep splash screen timing consistent
      setTimeout(() => {
        router.push("/login");
        // Start fade out after navigation begins
        setTimeout(() => {
          setLogoutSplashFadeOut(true);
        }, 1000);
        // Splash will be hidden by the login page when it mounts
      }, 2000);
    } finally {
      setIsLoggingOut(false);
    }
  };

  const hideLogoutSplash = () => {
    setShowLogoutSplash(false);
    setLogoutSplashFadeOut(false);
  };

  React.useEffect(() => {
    if (typeof window === "undefined") return;

    // Check for legacy token first
    const legacyToken = getToken();
    const savedEmail = Cookies.get('username');
    if (legacyToken) {
      // Legacy authentication
      setIsAuthenticated(true);
      if (savedEmail) {
        setUserEmail(savedEmail);
      }
      return;
    }

    // Check for secure session
    hasValidSession().then((valid) => {
      if (valid) {
        setIsAuthenticated(true);
        if (savedEmail) {
          setUserEmail(savedEmail);
        }
      } else {
        setIsAuthenticated(false);
        // Only redirect to login if not already on login page
        if (typeof window !== "undefined" && window.location.pathname !== "/login") {
          router.replace("/login");
        }
      }
    });
  }, []);

  const value = React.useMemo(
    () => ({
      isAuthenticated,
      login,
      logout,
      isLoggingOut,
      showLogoutSplash,
      logoutSplashFadeOut,
      userEmail,
      hideLogoutSplash,
    }),
    [isAuthenticated, isLoggingOut, showLogoutSplash, logoutSplashFadeOut, userEmail]
  );

  return (
    <AuthContext.Provider value={value}>
      {children}
    </AuthContext.Provider>
  );
}

/**
 * Hook to access authentication context
 */
export function useAuthContext() {
  const context = React.useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuthContext must be used within an AuthProvider');
  }
  return context;
}
