import { useCallback } from 'react';
import { api } from '@/lib/api';
import type Device from '@/app/types/Device';
import { useApiData, useApiSingleData } from '@/shared/hooks';
import type { UseApiDataOptions } from '@/shared/hooks';

/**
 * Options for the useDevices hook
 */
interface UseDevicesOptions extends Omit<UseApiDataOptions<Device[]>, 'initialData'> {
  /** Auto-fetch devices on mount */
  autoFetch?: boolean;
  /** Refetch interval in milliseconds (0 = no polling) */
  refetchInterval?: number;
  /** Network ID to filter devices by */
  networkId?: string;
}

/**
 * useDevices Hook
 *
 * A convenient hook for managing devices with common patterns:
 * - Fetching devices list
 * - Creating new devices
 * - Deleting devices
 * - Filtering by network
 * - Auto-refresh/polling
 * - Loading and error states
 *
 *
 * @param options - Configuration options
 * @returns Device management utilities
 *
 */
export const useDevices = (options: UseDevicesOptions = {}) => {
  const { autoFetch = true, refetchInterval = 0, networkId, ...restOptions } = options;

  // Create fetch function that handles networkId filtering
  const fetchDevicesFn = useCallback(async () => {
    if (networkId) {
      // Fetch devices for a specific network
      return api.getClient().get<Device[]>(`/networks/${networkId}/devices`);
    } else {
      // Fetch all devices
      return api.devices.getDevices();
    }
  }, [networkId]);

  // Use the generic useApiData hook
  const {
    data: devices,
    loading,
    error,
    refetch,
    addItem,
    removeItem,
    clearError,
  } = useApiData(
    fetchDevicesFn,
    {
      autoFetch,
      refetchInterval,
      initialData: [],
      enableOptimisticUpdates: true,
      ...restOptions,
    }
  );

  /**
   * Fetch a single device by ID
   */
  const getDevice = useCallback(async (id: string): Promise<Device | null> => {
    try {
      const device = await api.devices.getDevice(id);
      return device || null;
    } catch (err) {
      console.error('Failed to fetch device:', err);
      return null;
    }
  }, []);

  /**
   * Create a new device
   */
  const createDevice = useCallback(async (deviceData: {
    name: string;
    type: string;
    networkId?: string;
  }): Promise<Device | null> => {
    try {
      const { networkId: targetNetworkId, ...data } = deviceData;
      const endpoint = targetNetworkId
        ? `/networks/${targetNetworkId}/devices`
        : '/devices';

      const device = await api.getClient().post<Device>(endpoint, data);

      // Optimistically add to local state
      if (device) {
        addItem(device);
      }

      return device || null;
    } catch (err) {
      console.error('Failed to create device:', err);
      throw err;
    }
  }, [addItem]);

  /**
   * Delete a device
   */
  const deleteDevice = useCallback(async (id: string): Promise<boolean> => {
    try {
      await api.devices.deleteDevice(id);

      // Optimistically remove from local state
      removeItem(id);

      return true;
    } catch (err) {
      console.error('Failed to delete device:', err);
      throw err;
    }
  }, [removeItem]);

  return {
    // State
    devices,
    loading,
    error,

    // Actions
    fetchDevices: refetch,
    refetch,
    getDevice,
    createDevice,
    deleteDevice,
    clearError,
  };
};

/**
 * useDevice Hook (for single device)
 *
 * Simplified hook for managing a single device
 *
 * Now built on top of useApiSingleData for consistency.
 *
 * @param deviceId - The ID of the device to fetch
 * @param options - Configuration options
 * @returns Single device management utilities
 *
 * @example
 * ```tsx
 * const { device, loading, error, refetch, remove } = useDevice('device-123');
 *
 * if (loading) return <Spinner />;
 * if (error) return <Error message={error.message} />;
 * if (!device) return <NotFound />;
 *
 * return <DeviceDetails device={device} onDelete={remove} />;
 * ```
 */
export const useDevice = (
  deviceId: string | null,
  options: { autoFetch?: boolean } = {}
) => {
  const { autoFetch = true } = options;

  // Use the generic useApiSingleData hook
  const {
    data: device,
    loading,
    error,
    refetch,
  } = useApiSingleData(
    (id) => api.getClient().get<Device>(`/devices/${id}`),
    deviceId,
    {
      autoFetch,
      initialData: null,
    }
  );

  /**
   * Delete the device
   */
  const deleteDevice = useCallback(async (): Promise<boolean> => {
    if (!deviceId) return false;

    try {
      await api.getClient().delete(`/devices/${deviceId}`);
      return true;
    } catch (err) {
      console.error('Failed to delete device:', err);
      throw err;
    }
  }, [deviceId]);

  return {
    // State
    device,
    loading,
    error,

    // Actions
    refetch,
    remove: deleteDevice,
  };
};
