import { useCallback } from 'react';
import { api } from '@/lib/api';
import type Network from '@/app/types/Network';
import { useApiData, useApiSingleData } from '@/shared/hooks';
import type { UseApiDataOptions } from '@/shared/hooks';

/**
 * Options for the useNetworks hook
 */
interface UseNetworksOptions extends Omit<UseApiDataOptions<Network[]>, 'initialData'> {
  /** Auto-fetch networks on mount */
  autoFetch?: boolean;
  /** Refetch interval in milliseconds (0 = no polling) */
  refetchInterval?: number;
}

/**
 * useNetworks Hook
 *
 * A convenient hook for managing networks with common patterns:
 * - Fetching networks list
 * - Creating new networks
 * - Updating networks
 * - Deleting networks
 * - Auto-refresh/polling
 * - Loading and error states
 *
 * Now built on top of useApiData for consistency and reduced duplication.
 *
 * @param options - Configuration options
 * @returns Network management utilities
 *
 */
export const useNetworks = (options: UseNetworksOptions = {}) => {
  const { autoFetch = true, refetchInterval = 0, ...restOptions } = options;

  // Use the generic useApiData hook
  const {
    data: networks,
    loading,
    error,
    refetch,
    addItem,
    removeItem,
    clearError,
  } = useApiData(
    () => api.networks.getNetworks(),
    {
      autoFetch,
      refetchInterval,
      initialData: [],
      enableOptimisticUpdates: true,
      ...restOptions,
    }
  );

  /**
   * Fetch a single network by ID
   */
  const getNetwork = useCallback(async (id: string): Promise<Network | null> => {
    try {
      const network = await api.networks.getNetwork(id);
      return network || null;
    } catch (err) {
      console.error('Failed to fetch network:', err);
      return null;
    }
  }, []);

  /**
   * Create a new network
   */
  const createNetwork = useCallback(async (networkData: {
    name: string;
    type: string;
  }): Promise<Network | null> => {
    try {
      const network = await api.networks.createNetwork(networkData);

      // Optimistically add to local state
      if (network) {
        addItem(network);
      }

      return network || null;
    } catch (err) {
      console.error('Failed to create network:', err);
      throw err;
    }
  }, [addItem]);

  /**
   * Delete a network
   */
  const deleteNetwork = useCallback(async (id: string): Promise<boolean> => {
    try {
      await api.networks.deleteNetwork(id);

      // Optimistically remove from local state
      removeItem(id);

      return true;
    } catch (err) {
      console.error('Failed to delete network:', err);
      throw err;
    }
  }, [removeItem]);

  return {
    // State
    networks,
    loading,
    error,

    // Actions
    fetchNetworks: refetch,
    refetch,
    getNetwork,
    createNetwork,
    deleteNetwork,
    clearError,
  };
};

/**
 *
 * hook for managing a single network
 *
 *
 * @param networkId - The ID of the network to fetch
 * @param options - Configuration options
 * @returns Single network management utilities
 *
 */
export const useNetwork = (
  networkId: string | null,
  options: { autoFetch?: boolean; refetchInterval?: number } = {}
) => {
  const { autoFetch = true, refetchInterval = 0 } = options;

  // Use the generic useApiSingleData hook
  const {
    data: network,
    loading,
    error,
    refetch,
  } = useApiSingleData(
    (id) => api.networks.getNetwork(id),
    networkId,
    {
      autoFetch,
      refetchInterval,
      initialData: null,
    }
  );

  /**
   * Delete the network
   */
  const deleteNetwork = useCallback(async (): Promise<boolean> => {
    if (!networkId) return false;

    try {
      await api.networks.deleteNetwork(networkId);
      return true;
    } catch (err) {
      console.error('Failed to delete network:', err);
      throw err;
    }
  }, [networkId]);

  return {
    // State
    network,
    loading,
    error,

    // Actions
    refetch,
    remove: deleteNetwork,
  };
};
