/**
 * PDFMapEditor - Refactored
 * Main component for PDF-based network map editing
 *
 */

import React, { useState, useReducer, useRef, useCallback, useEffect, useMemo } from 'react';
import {
  EuiPage,
  EuiPageBody,
  EuiPageHeader,
  EuiTitle,
} from '@elastic/eui';
import { ErrorBoundary } from '@/shared/components';
import { useDragAndDrop } from '@/shared/hooks';
import { useLoader } from '@/shared/providers';
import SetupFlowManager from '@/features/project-setup/SetupFlowManager';
import { useScreenManager } from '@/features/project-setup/hooks/useScreenManager';
import { SCREEN_TYPES } from '@/features/project-setup/utils/constants';
import { useProjects } from '@/features/projects';
import { useAccessPoints } from '@/features/access-points';
import { usePDFLoader } from '@/features/pdf-editor';
import {
  migrateToRelativeCoordinates,
  relativeToPixel,
  getCurrentImageDimensions,
} from '@/lib/nodeScaling';
import { TIMING } from '@/lib/uiConstants';
import { AccessPointMarkerPoint } from '@/features/access-points';
import { useResponsiveLayout } from '@/shared/hooks';
import { viewportReducer, initialState } from '../../../../store/reducer';
import { api } from '@/lib/api';

// Custom hooks
import { usePDFMapState } from '@/features/pdf-editor';
import { usePDFMapActions } from '@/features/pdf-editor';
import { useViewportControls } from '@/features/pdf-editor';
import { useToast } from "@/components/ToastProvider";

// Screen components
import { UploadScreen } from '@/app/components/pdf-editor/screens/UploadScreen';
import { PDFCropScreen } from '@/app/components/pdf-editor/screens/PDFCropScreen';
import { MainProjectSetupScreen as MainProjectSetupScreen } from '@/app/components/pdf-editor/screens/MainProjectSetupScreen';
import { ModalsContainer } from '@/app/components/pdf-editor/screens/ModalsContainer';
import { LoadingScreens } from '@/features/project-setup/components/screens/LoadingScreens';
import { MultiStepDemo } from '@/features/project-setup/components/ui';
import AP from "@/app/types/AP";

interface PDFMapEditorProps {
    aps: AP[];
}

export default function PDFMapEditor({ aps }: PDFMapEditorProps) {
  // Reducer state for viewport
  const [state, dispatch] = useReducer(viewportReducer, initialState);

  // Context and hooks
  const { generateDeviceDetails } = useAccessPoints(aps);
  const { pdfJsLoaded, loadingError, setLoadingError, isClient } = usePDFLoader();
  const {
    projects: savedProjects,
    refetch: loadProjectsFromApi,
    createProject,
    loading: projectsLoading
  } = useProjects({
    autoFetch: false, // We'll manually control when to fetch
  });
  const { addToast } = useToast();
  const { globalAps, completeCurrentStep, setCurrentStep, currentScreen } = useLoader();
  const { navigateTo } = useScreenManager();
  const { isMobile } = useResponsiveLayout();

  // Access points state
  const [accessPoints, setAccessPoints] = useState<AccessPointMarkerPoint[]>([]);
  const accessPointTypes = globalAps;

  // Navigation state
  const [goToStep, setgoToStep] = useState(SCREEN_TYPES.NETWORK_SETUP);
  const [forceAllowMovement, setForceAllowMovement] = useState(false);

  // Local state for saved flag (replaces old useProjectManager.saved)
  const [saved, setSaved] = useState(false);
  const resetSaved = () => setSaved(false);

  // Wrap createProject to set saved flag on success
  const saveProjectToApi = useCallback(async (data: Record<string, unknown>) => {
    try {
      const result = await createProject(data);
      setSaved(true); // Set saved flag on success
      return result;
    } catch (error) {
      throw error; // Re-throw to let caller handle
    }
  }, [createProject]);

  // Load the first project function (replaces useProjectLoader)
  const loadFirstProject = useCallback(async () => {
    try {
      // Use API client to get first project
      interface LoadedProjectResponse {
        project?: {
          accessPoints: AccessPointMarkerPoint[];
          currentPage?: number;
          pageRotation?: number;
          viewportState?: {
            pageViewports?: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>;
          };
          pdfFile: {
            uploadedPath: string;
            name: string;
            originalName?: string;
          };
        };
        hasAccessPoints?: boolean;
      }

      const response = await api.getClient().get<LoadedProjectResponse>('/projects/first/load');

      if (!response.project) {
        return null;
      }

      // Load PDF blob if available
      let pdfBlob: Blob | null = null;
      let pdfFileName: string | null = null;

      if (response.project.pdfFile?.uploadedPath) {
        try {
          // Fetch the blob directly
          const blobResponse = await fetch(`/api${response.project.pdfFile.uploadedPath}`);
          pdfBlob = await blobResponse.blob();
          pdfFileName = response.project.pdfFile.originalName ||
                        response.project.pdfFile.name ||
                        'project.pdf';
        } catch (pdfError) {
          // Don't fail the entire load if PDF fetch fails
          const errorMessage = pdfError instanceof Error ? pdfError.message : 'Unknown error';
          console.error(`Could not load PDF file: ${errorMessage}`);
        }
      }

      return {
        project: response.project,
        pdfBlob,
        pdfFileName,
      };
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      console.error(`Could not load first project from backend: ${errorMessage}`);
      addToast({
        title: 'Error',
        color: 'danger',
        text: 'Failed to load project'
      });
      return null;
    }
  }, [addToast]);

  // Consolidated state management hook
  const {
    pdfState,
    setPdfFile,
    setPdfPages,
    setCurrentPage,
    setPageViewports,
    projectState,
    setProjectName,
    setProjectFound,
    setProjectUpdated,
    uiState,
    setEditingNodeId,
    setEditingNodeName,
    setSelectedNodeDetails,
    setShowNodeDetailsModal,
    setShowSaveModal,
    setShowLoadModal,
    loadingState,
    setApiLoading,
    setLoading,
    setStatus,
    setIsLoading,
    setProgress,
  } = usePDFMapState();

  // Refs
  const mapContainerRef = useRef<HTMLDivElement | null>(null);
  const mainRef = useRef<HTMLDivElement | null>(null);
  const containerRef = useRef<HTMLDivElement | null>(null);
  const mapImageRef = useRef<HTMLImageElement | null>(null);

  // Extract values for easy access
  const { file: pdfFile, pages: pdfPages, currentPage, pageViewports } = pdfState;
  const { name: projectName, found: projectFound, updated: projectUpdated } = projectState;
  const {
    editingNodeId,
    editingNodeName,
    selectedNodeDetails,
    showNodeDetailsModal,
    showSaveModal,
    showLoadModal
  } = uiState;
  const { apiLoading, loading, status, isLoading, progress } = loadingState;

  // Calculate allow node movement
  const allowNodeMovement = useMemo(() => {
    if (forceAllowMovement) return true;
    return !(pdfPages.length > 0 && projectFound);
  }, [pdfPages.length, projectFound, forceAllowMovement]);

  // Drag and drop hook
  const {
    draggedItem,
    touchDragItem,
    touchDragPreview,
    handleDragStart,
    handleTouchStartType,
    handleAccessPointDragStart,
    handleTouchStartAccessPoint,
    handleDrop,
    handleDragOver,
    handleTouchMove,
    handleTouchEnd,
    setLimitMessage
  } = useDragAndDrop({
    accessPoints,
    setAccessPoints,
    currentPage,
    allowNodeMovement,
    accessPointTypes,
    generateDeviceDetails,
    mapContainerRef,
    mapImageRef,
    state
  });

  // Actions hook
  const {
    handleFileUpload,
    saveProject,
    loadProject,
  } = usePDFMapActions({
    pdfFile,
    setPdfFile,
    setPdfPages,
    setCurrentPage,
    setAccessPoints,
    setProjectName,
    setProjectFound,
    setProjectUpdated,
    setShowSaveModal,
    setShowLoadModal,
    setApiLoading,
    setLoading,
    setStatus,
    setLoadingError,
    setPageViewports,
    addToast,
    pdfJsLoaded,
    projectName,
    currentPage,
    accessPoints,
    pageViewports,
    viewport: state.viewport,
    saveProjectToApi,
    generateDeviceDetails,
    mapImageRef,
    imageDimensions: state.imageDimensions,
    dispatch,
    setLimitMessage,
    completeCurrentStep,
    setCurrentStep,
    navigateTo,
    savedProjects
  });

  // Viewport controls hook
  const { handleWheel, handleMouseDown } = useViewportControls({
    viewport: state.viewport,
    dispatch,
    mapContainerRef,
    isMobile
  });

  // Update image dimensions when the image loads
  const handleImageLoad = useCallback(() => {
    if (mapImageRef.current) {
      const { offsetWidth, offsetHeight } = mapImageRef.current;
      dispatch({
        type: 'SET_IMAGE_DIMENSIONS',
        payload: { width: offsetWidth, height: offsetHeight }
      });
      // Migrate existing access points to relative coordinates
      setAccessPoints(prevPoints =>
        prevPoints.map(ap => migrateToRelativeCoordinates(ap, state.imageDimensions.width, state.imageDimensions.height))
      );
    }
  }, [state.imageDimensions.width, state.imageDimensions.height]);

  // Handle window resize
  useEffect(() => {
    const handleResize = () => {
      if (mapImageRef.current) {
        const { width, height } = mapImageRef.current;
        dispatch({
          type: 'SET_IMAGE_DIMENSIONS',
          payload: { width, height }
        });
      }
    };

    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  // Load saved projects on mount
  useEffect(() => {
    const loadSavedProjects = async () => {
      if (!isClient || !pdfJsLoaded) return;

      try {
        setStatus(true);
        const result = await loadFirstProject();

        if (!result?.project) {
          setProjectFound(false);
          setStatus(false);
          return;
        }

        await loadProjectsFromApi();

        if (result.project.pageRotation) {
          dispatch({ type: 'SET_ROTATION', payload: result.project.pageRotation });
        }

        const accessPointsWithDetails = result.project.accessPoints.map((ap: AccessPointMarkerPoint) => ({
          ...ap,
          deviceDetails: generateDeviceDetails(ap.type)
        }));

        setAccessPoints(accessPointsWithDetails);
        setCurrentPage(result.project.currentPage || 0);

        if (result.project.viewportState) {
          if (result.project.viewportState.pageViewports) {
            setPageViewports(result.project.viewportState.pageViewports);
          }

          const currentPageViewport = result.project.viewportState.pageViewports?.[result.project.currentPage || 0];
          if (currentPageViewport) {
            dispatch({ type: 'SET_ZOOM', payload: currentPageViewport.zoomLevel });
            dispatch({ type: 'SET_PAN', payload: currentPageViewport.panOffset });
          }
        }

        if (result.pdfBlob && result.pdfFileName) {
          const { processPdfFromFile } = await import('@/features/pdf-editor/utils/pdfUtils');
          const pdfFileWithPath = Object.assign(
            new File([result.pdfBlob], result.pdfFileName, {
              type: 'application/pdf'
            }),
            { uploadedPath: result.project.pdfFile.uploadedPath }
          );

          const pages = await processPdfFromFile(pdfFileWithPath);
          setPdfPages(pages);
          setPdfFile(pdfFileWithPath);
          setProjectFound(true);
        }

        setStatus(false);
      } catch (error) {
        setStatus(false);
        const { handleError } = await import('@/lib/errorHandler');
        handleError(error, { addToast, showToast: true, setLoadingError });
      }
    };

    loadSavedProjects();
  }, [isClient, pdfJsLoaded]);

  // Add touch event listeners for drag & drop
  useEffect(() => {
    const touchMoveHandler = (e: TouchEvent) => {
      if (touchDragItem) {
        e.preventDefault();
        handleTouchMove(e);
      }
    };

    const touchEndHandler = (e: TouchEvent) => {
      if (touchDragItem) {
        e.preventDefault();
        handleTouchEnd(e);
      }
    };

    document.addEventListener('touchmove', touchMoveHandler, { passive: false });
    document.addEventListener('touchend', touchEndHandler, { passive: false });

    return () => {
      document.removeEventListener('touchmove', touchMoveHandler);
      document.removeEventListener('touchend', touchEndHandler);
    };
  }, [handleTouchMove, handleTouchEnd, touchDragItem]);

  // Calculate current page access points
  const currentPageAccessPoints = useMemo(() => {
    const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);

    return accessPoints
      .filter(ap => ap.page === currentPage)
      .map(ap => {
        if (typeof ap.relativeX === 'number' && typeof ap.relativeY === 'number') {
          const { x, y } = relativeToPixel(
            ap.relativeX, ap.relativeY,
            currentDimensions.width, currentDimensions.height
          );
          return { ...ap, x, y };
        }
        return ap;
      });
  }, [accessPoints, currentPage, state.imageDimensions.width, state.imageDimensions.height]);

  // Start editing node name
  const startEditingNode = (accessPoint: AccessPointMarkerPoint) => {
    setEditingNodeId(accessPoint.id);
    setProjectUpdated(true);
    setEditingNodeName(accessPoint.floor);
  };

  // Show node details modal
  const showNodeDetails = (accessPoint: AccessPointMarkerPoint) => {
    setSelectedNodeDetails(accessPoint);
    setShowNodeDetailsModal(true);
  };

  // Initial loading timeout
  useEffect(() => {
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, TIMING.LOADING_TIMEOUT);
    return () => clearTimeout(timer);
  }, []);

  if (!pdfJsLoaded) {
    return <></>;
  }

  return (
    <EuiPage>
      <EuiPageBody>
        <EuiPageHeader>
          <EuiTitle size="l">
            <h1></h1>
          </EuiTitle>
        </EuiPageHeader>
        {(savedProjects.length === 0 && !isLoading) && <MultiStepDemo />}
        <SetupFlowManager
          setCurrentScreen={(step) => navigateTo(step)}
          status={status}
          savedProjects={savedProjects}
        />
        <ErrorBoundary>
          {/* Upload Screen */}
          {((currentScreen === SCREEN_TYPES.UPLOAD) && !projectFound) && (
            <UploadScreen
              currentScreen={currentScreen}
              loading={loading}
              pdfJsLoaded={pdfJsLoaded}
              loadingError={loadingError}
              onFileUpload={handleFileUpload}
            />
          )}

          {/* PDF Crop Editor */}
          {currentScreen === SCREEN_TYPES.PDF_CROP && (
            <PDFCropScreen
              imgsrc={pdfPages[currentPage]}
              navigateTo={navigateTo}
              dispatch={dispatch}
              setPdfFile={setPdfFile}
            />
          )}

          {/* Main Project Setup */}
          {currentScreen === SCREEN_TYPES.PROJECT_SETUP && (
            <MainProjectSetupScreen
              pdfPages={pdfPages}
              accessPointTypes={accessPointTypes}
              accessPoints={accessPoints}
              currentPageAccessPoints={currentPageAccessPoints}
              projectFound={projectFound}
              projectUpdated={projectUpdated}
              allowNodeMovement={allowNodeMovement}
              editingNodeId={editingNodeId}
              editingNodeName={editingNodeName}
              draggedItem={draggedItem}
              touchDragPreview={touchDragPreview}
              touchDragItem={touchDragItem}
              loading={loading}
              viewport={state.viewport}
              imageDimensions={state.imageDimensions}
              src={state.src}
              savedProjects={savedProjects}
              handleDragStart={handleDragStart}
              handleTouchStartType={handleTouchStartType}
              handleAccessPointDragStart={handleAccessPointDragStart}
              handleTouchStartAccessPoint={handleTouchStartAccessPoint}
              setEditingNodeName={setEditingNodeName}
              startEditingNode={startEditingNode}
              setAccessPoints={setAccessPoints}
              setEditingNodeId={setEditingNodeId}
              onSave={() => setShowSaveModal(true)}
              onToggleNodeMovement={setForceAllowMovement}
              handleImageLoad={handleImageLoad}
              handleDrop={handleDrop}
              handleDragOver={handleDragOver}
              handleWheel={handleWheel}
              handleMouseDown={handleMouseDown}
              showNodeDetails={showNodeDetails}
              dispatch={dispatch}
              containerRef={containerRef}
              mapContainerRef={mapContainerRef}
              mapImageRef={mapImageRef}
              mainRef={mainRef}
              onContinue={() => {
                setLoading(true);
                navigateTo(SCREEN_TYPES.PROJECT_MODAL);
                completeCurrentStep();
                setLoading(false);
              }}
            />
          )}

          {/* Modals */}
          <ModalsContainer
            showNodeDetailsModal={showNodeDetailsModal}
            selectedNodeDetails={selectedNodeDetails}
            showSaveModal={showSaveModal}
            showLoadModal={showLoadModal}
            projectName={projectName}
            apiLoading={apiLoading}
            pdfFile={pdfFile}
            savedProjects={savedProjects}
            accessPointTypes={accessPointTypes}
            globalAps={globalAps}
            setShowNodeDetailsModal={setShowNodeDetailsModal}
            setShowSaveModal={setShowSaveModal}
            setShowLoadModal={setShowLoadModal}
            setProjectName={setProjectName}
            startEditingNode={startEditingNode}
            saveProject={saveProject}
            loadProject={loadProject}
          />

          {/* Loading/Complete/Error Screens */}
          <LoadingScreens
            currentScreen={currentScreen}
            progress={progress}
            loadingError={loadingError}
            goToStep={goToStep}
            projectName={projectName}
            saved={saved}
            navigateTo={navigateTo}
            setProgress={setProgress}
            setgoToStep={setgoToStep}
            resetSaved={resetSaved}
            setProjectName={setProjectName}
            saveProject={saveProject}
            loadFirstProject={loadFirstProject}
            loadProjectsFromApi={loadProjectsFromApi}
            generateDeviceDetails={generateDeviceDetails}
            setAccessPoints={setAccessPoints}
            setCurrentPage={setCurrentPage}
            setPageViewports={setPageViewports}
            setPdfPages={setPdfPages}
            setPdfFile={setPdfFile}
            setProjectFound={setProjectFound}
            setStatus={setStatus}
            dispatch={dispatch}
            addToast={addToast}
            setLoadingError={setLoadingError}
          />
        </ErrorBoundary>
      </EuiPageBody>
    </EuiPage>
  );
}
