import { useState, useEffect } from 'react';
import { handleError, createPdfError } from '@/lib/errorHandler';

export const usePDFLoader = () => {
  const [pdfJsLoaded, setPdfJsLoaded] = useState(false);
  const [jsPdfLoaded, setJsPdfLoaded] = useState(false);
  const [loadingError, setLoadingError] = useState<string | null>(null);
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(typeof window !== 'undefined');
  }, []);

  useEffect(() => {
    let script: HTMLScriptElement | null = null;

    const loadPdfJs = async () => {
      if (!isClient) return;

      if (typeof window !== 'undefined' && !window.pdfjsLib) {
        try {
          script = document.createElement('script');
          script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js';
          script.onload = () => {
            if (window.pdfjsLib) {
              window.pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';
              setPdfJsLoaded(true);
            } else {
              const error = createPdfError('Failed to initialize PDF.js library');
              handleError(error, {
                setLoadingError,
                showToast: false
              });
            }
          };
          script.onerror = () => {
            const error = createPdfError('Failed to load PDF.js library from CDN');
            handleError(error, {
              setLoadingError,
              showToast: false
            });
          };
          document.head.appendChild(script);
        } catch (error) {
          handleError(
            createPdfError('Error loading PDF.js', error),
            { setLoadingError, showToast: false }
          );
        }
      } else {
        if (window.pdfjsLib) {
          setPdfJsLoaded(true);
        }
      }
    };

    loadPdfJs();

    return () => {
      if (script && document.head.contains(script)) {
        document.head.removeChild(script);
      }
    };
  }, [isClient]);

  useEffect(() => {
    let script: HTMLScriptElement | null = null;

    const loadJsPDF = async () => {
      if (!isClient) return;

      if (typeof window !== 'undefined' && !window.jsPDF) {
        try {
          script = document.createElement('script');
          script.src = 'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js';
          script.onload = () => {
            setJsPdfLoaded(true);
          };
          script.onerror = () => {
            const error = createPdfError('Failed to load jsPDF library from CDN');
            handleError(error, {
              setLoadingError,
              showToast: false
            });
          };
          document.head.appendChild(script);
        } catch (error) {
          handleError(
            createPdfError('Error loading jsPDF', error),
            { setLoadingError, showToast: false }
          );
        }
      } else {
        if (window.jsPDF) {
          setJsPdfLoaded(true);
        }
      }
    };

    loadJsPDF();

    return () => {
      if (script && document.head.contains(script)) {
        document.head.removeChild(script);
      }
    };
  }, [isClient]);

  return { pdfJsLoaded, jsPdfLoaded, loadingError, setLoadingError, isClient };
};
