/**
 * Custom hook for PDF Map Editor actions
 * Handles business logic for PDF upload, project save/load, and node management
 */

import { useCallback } from 'react';
import { PDFFile } from '@/app/components/pdf-editor/types';
import { AccessPointMarkerPoint } from '@/features/access-points';
import { processPdfFromFile, uploadPdfToBackend } from '@/features/pdf-editor/utils/pdfUtils';
import { handleError, createValidationError, createPdfError, AppError, ErrorCategory } from '@/lib/errorHandler';
import { validatePdfFile } from '@/lib/fileValidation';
import {
  migrateToRelativeCoordinates,
  getCurrentImageDimensions
} from '@/lib/nodeScaling';
import getDict from '@/lib/dict';
import type {
  ViewportState,
  ViewportAction,
} from '@/types';
import { DeviceDetails, ImageDimensions, Project, Toast } from "@/shared/types";

const dict = getDict("de");

interface ProjectData {
  name: string;
  accessPoints: Array<{
    id: string;
    type: string;
    name: string;
    floor: string;
    x: number;
    y: number;
    page: number;
  }>;
  currentPage: number;
  timestamp: number;
  pageRotation: number;
  pdfFile: {
    name: string;
    size: number;
    originalName: string;
    uploadedPath: string | null;
  };
  viewportState: {
    zoomLevel: number;
    panOffset: { x: number; y: number };
    pageViewports: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>;
  };
}

interface UsePDFMapActionsProps {
  pdfFile: PDFFile;
  setPdfFile: (file: PDFFile | ((prev: PDFFile) => PDFFile)) => void;
  setPdfPages: (pages: string[]) => void;
  setCurrentPage: (page: number) => void;
  setAccessPoints: (points: AccessPointMarkerPoint[] | ((prev: AccessPointMarkerPoint[]) => AccessPointMarkerPoint[])) => void;
  setProjectName: (name: string) => void;
  setProjectFound: (found: boolean) => void;
  setProjectUpdated: (updated: boolean) => void;
  setShowSaveModal: (show: boolean) => void;
  setShowLoadModal: (show: boolean) => void;
  setApiLoading: (loading: boolean) => void;
  setLoading: (loading: boolean) => void;
  setStatus: (status: boolean) => void;
  setLoadingError: (error: string | null) => void;
  setPageViewports: (viewports: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>) => void;
  addToast: (toast: Toast) => void;
  pdfJsLoaded: boolean;
  projectName: string;
  currentPage: number;
  accessPoints: AccessPointMarkerPoint[];
  pageViewports: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>;
  viewport: ViewportState;
  saveProjectToApi: (data: ProjectData) => Promise<Project | null>;
  generateDeviceDetails: (type: string) => DeviceDetails;
  mapImageRef: React.RefObject<HTMLImageElement>;
  imageDimensions: ImageDimensions;
  dispatch: React.Dispatch<ViewportAction>;
  setLimitMessage: (msg: string) => void;
  completeCurrentStep: () => void;
  setCurrentStep: (step: number | ((prev: number) => number)) => void;
  navigateTo: (screen: string) => void;
  savedProjects: Project[];
}

export function usePDFMapActions(props: UsePDFMapActionsProps) {
  const {
    pdfFile,
    setPdfFile,
    setPdfPages,
    setCurrentPage,
    setAccessPoints,
    setProjectName,
    setProjectFound,
    setProjectUpdated,
    setShowSaveModal,
    setShowLoadModal,
    setApiLoading,
    setLoading,
    setStatus,
    setLoadingError,
    setPageViewports,
    addToast,
    pdfJsLoaded,
    projectName,
    currentPage,
    accessPoints,
    pageViewports,
    viewport,
    saveProjectToApi,
    generateDeviceDetails,
    mapImageRef,
    imageDimensions,
    dispatch,
    setLimitMessage,
    completeCurrentStep,
    setCurrentStep,
    navigateTo,
    savedProjects
  } = props;

  // Handle PDF file upload
  const handleFileUpload = useCallback(async (files: File[]) => {
    const file = files[0];

    // Validate file
    const validation = validatePdfFile(file);
    if (!validation.valid) {
      const error = createValidationError(validation.error || 'Invalid file', 'file');
      handleError(error, { setLoadingError, showToast: false });
      return;
    }

    if (!pdfJsLoaded || !window.pdfjsLib) {
      setLoadingError('PDF.js is still loading. Please wait a moment and try again.');
      return;
    }

    setPdfFile(file);
    setLoadingError(null);
    setLimitMessage('');
    setApiLoading(true);
    setLoading(true);

    try {
      const pages = await processPdfFromFile(file);
      setPdfPages(pages);
      setCurrentPage(0);
      setAccessPoints([]);

      // Reset zoom and pan
      dispatch({ type: 'SET_ZOOM', payload: 1 });
      dispatch({ type: 'SET_PAN', payload: { x: 0, y: 0 } });

      if (savedProjects.length < 0) {
        completeCurrentStep();
        setCurrentStep((prev: number) => prev + 1);
      }

      navigateTo('PDF_CROP');
      setStatus(false);
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      const pdfError = createPdfError(`Error processing PDF: ${errorMessage}`, error);
      handleError(pdfError, { setLoadingError, showToast: false });
    } finally {
      setLoading(false);
    }
  }, [
    pdfJsLoaded,
    setPdfFile,
    setPdfPages,
    setCurrentPage,
    setAccessPoints,
    setLoadingError,
    setApiLoading,
    setLoading,
    setStatus,
    dispatch,
    setLimitMessage,
    completeCurrentStep,
    setCurrentStep,
    navigateTo,
    savedProjects
  ]);

  // Prepare viewport state for saving
  const prepareViewportState = useCallback(() => ({
    zoomLevel: viewport.zoomLevel,
    panOffset: viewport.panOffset,
    pageViewports: {
      ...pageViewports,
      [currentPage]: {
        zoomLevel: viewport.zoomLevel,
        panOffset: viewport.panOffset
      }
    }
  }), [viewport, pageViewports, currentPage]);

  // Prepare project data for saving
  const prepareProjectData = useCallback((name: string) => ({
    name: name.trim(),
    accessPoints: accessPoints.map(ap => ({
      id: ap.id,
      type: ap.type,
      name: ap.name,
      floor: ap.floor,
      x: ap.relativeX,
      y: ap.relativeY,
      page: ap.page,
    })),
    currentPage,
    timestamp: Date.now(),
    pageRotation: viewport.rotation || 0,
    pdfFile: {
      name: pdfFile.name,
      size: pdfFile.size,
      originalName: pdfFile.name,
      uploadedPath: pdfFile.uploadedPath || null
    },
    viewportState: prepareViewportState()
  }), [accessPoints, currentPage, pdfFile, viewport, prepareViewportState]);

  // Ensure PDF is uploaded before saving
  const ensurePdfUploaded = useCallback(async (projectData: ProjectData) => {
    console.log('ensurePdfUploaded - Current pdfFile:', {
      name: pdfFile?.name,
      uploadedPath: pdfFile?.uploadedPath,
      hasFile: !!pdfFile
    });

    if (pdfFile && !pdfFile.uploadedPath) {
      console.log('Uploading PDF to backend...');
      const uploadedPath = await uploadPdfToBackend(pdfFile);

      console.log('Upload result - uploadedPath:', uploadedPath);

      if (uploadedPath) {
        setStatus(true);
        projectData.pdfFile.uploadedPath = uploadedPath;
        setPdfFile((prev: PDFFile) => ({ ...prev, uploadedPath }));
        console.log('Updated projectData.pdfFile.uploadedPath to:', uploadedPath);
      } else {
        setStatus(false);
        addToast({
          title: 'Error',
          color: 'danger',
          text: 'Failed to upload PDF to backend',
        });
      }
    } else if (pdfFile?.uploadedPath) {
      console.log('PDF already uploaded, using existing path:', pdfFile.uploadedPath);
      projectData.pdfFile.uploadedPath = pdfFile.uploadedPath;
    }
  }, [pdfFile, setPdfFile, setStatus, addToast]);

  // Save project
  const saveProject = useCallback(async () => {
    // Validation
    if (!projectName.trim()) {
      const error = createValidationError('Please enter a project name', 'projectName');
      handleError(error, { setLoadingError, showToast: false });
      return;
    }

    if (!pdfFile) {
      const error = createValidationError('Please upload a PDF file before saving the project', 'pdfFile');
      handleError(error, { setLoadingError, showToast: false });
      return;
    }

    // Prepare and save
    const projectData = prepareProjectData(projectName);
    setApiLoading(true);

    try {
      await ensurePdfUploaded(projectData);
      await saveProjectToApi(projectData);

      // If we reach here, the project was saved successfully
      setShowSaveModal(false);
      setProjectName('');
      setProjectUpdated(true);
      addToast({
        title: '',
        color: "success",
        text: dict.projectSaved
      });
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      const appError = error instanceof AppError
        ? error
        : new AppError(`Error saving project: ${errorMessage}`, {
          category: ErrorCategory.PROJECT_MANAGEMENT,
          technicalDetails: error
        });
      handleError(appError, { setLoadingError, showToast: false });
    } finally {
      setApiLoading(false);
    }
  }, [
    projectName,
    pdfFile,
    accessPoints,
    currentPage,
    viewport,
    prepareProjectData,
    ensurePdfUploaded,
    saveProjectToApi,
    setApiLoading,
    setShowSaveModal,
    setProjectName,
    setProjectUpdated,
    addToast,
    setLoadingError
  ]);

  // Load project
  const loadProject = useCallback(async (project: Project) => {
    try {
      setLimitMessage('');

      const currentDimensions = getCurrentImageDimensions(mapImageRef, imageDimensions);
      const accessPointsWithDetails = (project.accessPoints || []).map((ap: AccessPointMarkerPoint) => {
        let migratedAp = {
          ...ap,
          deviceDetails: generateDeviceDetails(ap.type)
        };

        if (currentDimensions.width > 0 && currentDimensions.height > 0 &&
          (typeof migratedAp.relativeX !== 'number' || typeof migratedAp.relativeY !== 'number')) {
          migratedAp = migrateToRelativeCoordinates(migratedAp, currentDimensions.width, currentDimensions.height);
        }

        return migratedAp;
      });

      setAccessPoints(accessPointsWithDetails);
      setCurrentPage(project.currentPage || 0);

      if (project.viewportState) {
        if (project.viewportState.pageViewports) {
          setPageViewports(project.viewportState.pageViewports);
        }

        const currentPageViewport = project.viewportState.pageViewports?.[project.currentPage || 0];
        if (currentPageViewport) {
          dispatch({ type: 'SET_ZOOM', payload: currentPageViewport.zoomLevel });
          dispatch({ type: 'SET_PAN', payload: currentPageViewport.panOffset });
        } else if (project.viewportState.zoomLevel && project.viewportState.panOffset) {
          dispatch({ type: 'SET_ZOOM', payload: project.viewportState.zoomLevel });
          dispatch({ type: 'SET_PAN', payload: project.viewportState.panOffset });
        }
      } else {
        dispatch({ type: 'SET_ZOOM', payload: 1 });
        dispatch({ type: 'SET_PAN', payload: { x: 0, y: 0 } });
        setPageViewports({});
      }

      setShowLoadModal(false);

      // Load PDF from backend
      if (project.pdfFile?.uploadedPath) {
        try {
          setApiLoading(true);

          const pdfResponse = await fetch(`/api/${project.pdfFile.uploadedPath}`);
          if (pdfResponse.ok) {
            const pdfBlob = await pdfResponse.blob();
            const pdfFileWithPath = Object.assign(
              new File([pdfBlob], project.pdfFile.originalName || project.pdfFile.name, {
                type: 'application/pdf'
              }),
              { uploadedPath: project.pdfFile.uploadedPath }
            ) as PDFFile;

            const pages = await processPdfFromFile(pdfFileWithPath);
            setPdfPages(pages);
            setPdfFile(pdfFileWithPath);
            addToast({
              title: '',
              color: "success",
              text: dict.projectFound,
            });
          } else {
            addToast({
              title: 'Error',
              color: "danger",
              text: `Failed to load PDF: ${pdfResponse.status}`,
            });
          }
        } catch (pdfError) {
          setLoadingError(`Project loaded, but could not load PDF: ${project.pdfFile?.originalName || project.pdfFile?.name || 'Unknown'}. Please re-upload the PDF file.`);
        } finally {
          setApiLoading(false);
        }
      } else {
        setLoadingError(`Project "${project.name}" loaded successfully. Please upload a PDF file to continue editing.`);
      }

    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      const appError = new AppError(`Error loading project: ${errorMessage}`, {
        category: ErrorCategory.PROJECT_MANAGEMENT,
        technicalDetails: error
      });
      handleError(appError, { setLoadingError, showToast: false });
    }
  }, [
    setLimitMessage,
    setAccessPoints,
    setCurrentPage,
    setPageViewports,
    setShowLoadModal,
    setApiLoading,
    setPdfPages,
    setPdfFile,
    setLoadingError,
    addToast,
    generateDeviceDetails,
    mapImageRef,
    imageDimensions,
    dispatch
  ]);

  return {
    handleFileUpload,
    saveProject,
    loadProject,
    prepareViewportState,
    prepareProjectData,
  };
}
