
import { useState, useCallback } from 'react';
import { PDFFile } from '@/app/components/pdf-editor/types';

export interface PDFState {
  file: PDFFile;
  pages: string[];
  currentPage: number;
  pageViewports: Record<number, any>;
}

export interface ProjectState {
  name: string;
  found: boolean;
  updated: boolean;
  isNew: boolean;
}

export interface UIState {
  editingNodeId: string | null;
  editingNodeName: string;
  selectedNodeDetails: any | null;
  showNodeDetailsModal: boolean;
  showSaveModal: boolean;
  showLoadModal: boolean;
}

export interface LoadingState {
  apiLoading: boolean;
  loading: boolean;
  isGenerating: boolean;
  status: boolean;
  isLoading: boolean;
  progress: number;
}

export function usePDFMapState() {
  // PDF state
  const [pdfState, setPdfState] = useState<PDFState>({
    file: {
      uploadedPath: '',
      name: '',
      size: 0,
    } as PDFFile,
    pages: [],
    currentPage: 0,
    pageViewports: {},
  });

  // Project state
  const [projectState, setProjectState] = useState<ProjectState>({
    name: '',
    found: false,
    updated: false,
    isNew: false,
  });

  // UI state
  const [uiState, setUiState] = useState<UIState>({
    editingNodeId: null,
    editingNodeName: '',
    selectedNodeDetails: null,
    showNodeDetailsModal: false,
    showSaveModal: false,
    showLoadModal: false,
  });

  // Loading state
  const [loadingState, setLoadingState] = useState<LoadingState>({
    apiLoading: false,
    loading: false,
    isGenerating: false,
    status: false,
    isLoading: true,
    progress: 0,
  });

  // PDF state setters
  const setPdfFile = useCallback((fileOrUpdater: PDFFile | ((prev: PDFFile) => PDFFile)) => {
    setPdfState(prev => ({
      ...prev,
      file: typeof fileOrUpdater === 'function' ? fileOrUpdater(prev.file) : fileOrUpdater
    }));
  }, []);

  const setPdfPages = useCallback((pages: string[]) => {
    setPdfState(prev => ({ ...prev, pages }));
  }, []);

  const setCurrentPage = useCallback((page: number) => {
    setPdfState(prev => ({ ...prev, currentPage: page }));
  }, []);

  const setPageViewports = useCallback((viewports: Record<number, any>) => {
    setPdfState(prev => ({ ...prev, pageViewports: viewports }));
  }, []);

  // Project state setters
  const setProjectName = useCallback((name: string) => {
    setProjectState(prev => ({ ...prev, name }));
  }, []);

  const setProjectFound = useCallback((found: boolean) => {
    setProjectState(prev => ({ ...prev, found }));
  }, []);

  const setProjectUpdated = useCallback((updated: boolean) => {
    setProjectState(prev => ({ ...prev, updated }));
  }, []);

  const setIsNewProject = useCallback((isNew: boolean) => {
    setProjectState(prev => ({ ...prev, isNew }));
  }, []);

  // UI state setters
  const setEditingNodeId = useCallback((id: string | null) => {
    setUiState(prev => ({ ...prev, editingNodeId: id }));
  }, []);

  const setEditingNodeName = useCallback((name: string) => {
    setUiState(prev => ({ ...prev, editingNodeName: name }));
  }, []);

  const setSelectedNodeDetails = useCallback((details: any | null) => {
    setUiState(prev => ({ ...prev, selectedNodeDetails: details }));
  }, []);

  const setShowNodeDetailsModal = useCallback((show: boolean) => {
    setUiState(prev => ({ ...prev, showNodeDetailsModal: show }));
  }, []);

  const setShowSaveModal = useCallback((show: boolean) => {
    setUiState(prev => ({ ...prev, showSaveModal: show }));
  }, []);

  const setShowLoadModal = useCallback((show: boolean) => {
    setUiState(prev => ({ ...prev, showLoadModal: show }));
  }, []);

  // Loading state setters
  const setApiLoading = useCallback((loading: boolean) => {
    setLoadingState(prev => ({ ...prev, apiLoading: loading }));
  }, []);

  const setLoading = useCallback((loading: boolean) => {
    setLoadingState(prev => ({ ...prev, loading }));
  }, []);

  const setStatus = useCallback((status: boolean) => {
    setLoadingState(prev => ({ ...prev, status }));
  }, []);

  const setIsLoading = useCallback((loading: boolean) => {
    setLoadingState(prev => ({ ...prev, isLoading: loading }));
  }, []);

  const setProgress = useCallback((progress: number | ((prev: number) => number)) => {
    setLoadingState(prev => ({
      ...prev,
      progress: typeof progress === 'function' ? progress(prev.progress) : progress
    }));
  }, []);

  return {
    // PDF state
    pdfState,
    setPdfFile,
    setPdfPages,
    setCurrentPage,
    setPageViewports,

    // Project state
    projectState,
    setProjectName,
    setProjectFound,
    setProjectUpdated,
    setIsNewProject,

    // UI state
    uiState,
    setEditingNodeId,
    setEditingNodeName,
    setSelectedNodeDetails,
    setShowNodeDetailsModal,
    setShowSaveModal,
    setShowLoadModal,

    // Loading state
    loadingState,
    setApiLoading,
    setLoading,
    setStatus,
    setIsLoading,
    setProgress,
  };
}
