/**
 * Custom hook for viewport controls (zoom, pan, rotation)
 * Handles mouse and touch interactions for map navigation
 */

import { useCallback, useEffect, useState } from 'react';
import { getTouchDistance, MIN_ZOOM, MAX_ZOOM, ZOOM_STEP } from '@/lib/nodeScaling';

interface ViewportState {
  zoomLevel: number;
  panOffset: { x: number; y: number };
  isPanning: boolean;
  lastPanPosition: { x: number; y: number };
  rotation: number;
}

interface UseViewportControlsProps {
  viewport: ViewportState;
  dispatch: any;
  mapContainerRef: React.RefObject<HTMLDivElement>;
  isMobile: boolean;
}

export function useViewportControls({
  viewport,
  dispatch,
  mapContainerRef,
  isMobile
}: UseViewportControlsProps) {
  // Touch zoom/pan state
  const [touchStart, setTouchStart] = useState<any>(null);
  const [initialDistance, setInitialDistance] = useState<number | null>(null);
  const [initialZoom, setInitialZoom] = useState(1);
  const [isTouchPanning, setIsTouchPanning] = useState(false);
  const [lastTouchPanPosition, setLastTouchPanPosition] = useState({ x: 0, y: 0 });

  // Unified pan handler
  const handlePan = useCallback((deltaX: number, deltaY: number) => {
    dispatch({
      type: 'SET_PAN',
      payload: {
        x: viewport.panOffset.x + deltaX,
        y: viewport.panOffset.y + deltaY
      }
    });
  }, [viewport.panOffset, dispatch]);

  // Mouse wheel zoom
  const handleWheel = useCallback((e: React.WheelEvent) => {
    if (e.ctrlKey || e.metaKey) {
      e.preventDefault();
      const delta = e.deltaY > 0 ? -ZOOM_STEP : ZOOM_STEP;
      dispatch({
        type: 'SET_ZOOM',
        payload: Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, viewport.zoomLevel + delta))
      });
    }
  }, [viewport.zoomLevel, dispatch]);

  // Mouse pan handlers
  const handleMouseDown = useCallback((e: React.MouseEvent) => {
    if (e.button === 0 || e.button === 1) {
      dispatch({
        type: 'START_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
      e.preventDefault();
    }
  }, [dispatch]);

  const handleMouseMove = useCallback((e: MouseEvent) => {
    if (viewport.isPanning) {
      const deltaX = e.clientX - viewport.lastPanPosition.x;
      const deltaY = e.clientY - viewport.lastPanPosition.y;

      handlePan(deltaX, deltaY);

      dispatch({
        type: 'UPDATE_PAN',
        payload: { x: e.clientX, y: e.clientY }
      });
    }
  }, [viewport.isPanning, viewport.lastPanPosition, handlePan, dispatch]);

  const handleMouseUp = useCallback(() => {
    dispatch({ type: 'END_PAN' });
  }, [dispatch]);

  // Touch handlers for mobile zoom/pan
  const handleMobileTouchStart = useCallback((e: TouchEvent) => {
    if (e.touches.length === 2) {
      e.preventDefault();
      const distance = getTouchDistance(e.touches[0], e.touches[1]);
      setInitialDistance(distance);
      setInitialZoom(viewport.zoomLevel);
      setTouchStart({
        x1: e.touches[0].clientX,
        y1: e.touches[0].clientY,
        x2: e.touches[1].clientX,
        y2: e.touches[1].clientY
      });
      setIsTouchPanning(false);
    } else if (e.touches.length === 1) {
      const touch = e.touches[0];
      setIsTouchPanning(true);
      setLastTouchPanPosition({ x: touch.clientX, y: touch.clientY });
      setInitialDistance(null);
    }
  }, [viewport.zoomLevel]);

  const handleMobileTouchMove = useCallback((e: TouchEvent) => {
    if (e.touches.length === 2 && initialDistance && touchStart) {
      e.preventDefault();
      const currentDistance = getTouchDistance(e.touches[0], e.touches[1]);
      const scale = currentDistance / initialDistance;
      const newZoom = Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, initialZoom * scale));
      dispatch({ type: 'SET_ZOOM', payload: newZoom });
    } else if (e.touches.length === 1 && isTouchPanning) {
      e.preventDefault();
      const touch = e.touches[0];
      const deltaX = touch.clientX - lastTouchPanPosition.x;
      const deltaY = touch.clientY - lastTouchPanPosition.y;

      handlePan(deltaX, deltaY);

      setLastTouchPanPosition({ x: touch.clientX, y: touch.clientY });
    }
  }, [initialDistance, touchStart, initialZoom, isTouchPanning, lastTouchPanPosition, handlePan, dispatch]);

  const handleMobileTouchEnd = useCallback((e: TouchEvent) => {
    if (e.touches.length < 2) {
      setInitialDistance(null);
      setTouchStart(null);
    }
    if (e.touches.length === 0) {
      setIsTouchPanning(false);
    }
  }, []);

  // Add mouse event listeners
  useEffect(() => {
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);
    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [handleMouseMove, handleMouseUp]);

  // Add touch event listeners for mobile zoom/pan
  useEffect(() => {
    if (!isMobile) return;

    const mapContainer = mapContainerRef.current;
    if (mapContainer) {
      mapContainer.addEventListener('touchstart', handleMobileTouchStart, { passive: false });
      mapContainer.addEventListener('touchmove', handleMobileTouchMove, { passive: false });
      mapContainer.addEventListener('touchend', handleMobileTouchEnd, { passive: false });
    }

    return () => {
      if (mapContainer) {
        mapContainer.removeEventListener('touchstart', handleMobileTouchStart);
        mapContainer.removeEventListener('touchmove', handleMobileTouchMove);
        mapContainer.removeEventListener('touchend', handleMobileTouchEnd);
      }
    };
  }, [handleMobileTouchStart, handleMobileTouchMove, handleMobileTouchEnd, isMobile]);

  return {
    handleWheel,
    handleMouseDown,
    handlePan,
  };
}
