import { handleError, createPdfError, createNetworkError, AppError } from '@/lib/errorHandler';
import { api, APIError } from '@/lib/api';

export const processPdfFromFile = async (file) => {
  try {
    const arrayBuffer = await file.arrayBuffer();
    const pdf = await window.pdfjsLib.getDocument({ data: arrayBuffer }).promise;

    const pages = [];

    for (let i = 1; i <= pdf.numPages; i++) {
      const page = await pdf.getPage(i);
      const viewport = page.getViewport({ scale: 2 });
      const canvas = document.createElement('canvas');
      const context = canvas.getContext('2d');
      canvas.height = viewport.height;
      canvas.width = viewport.width;

      await page.render({
        canvasContext: context,
        viewport: viewport
      }).promise;

      pages.push(canvas.toDataURL('image/png'));
    }

    return pages;
  } catch (error) {
    const appError = createPdfError(`Failed to process PDF: ${error.message}`, error);
    handleError(appError, { showToast: false });
    throw appError;
  }
};

export const uploadPdfToBackend = async (file) => {
  try {
    const pdfFile = file instanceof File ? file : new File([file], 'document.pdf', { type: 'application/pdf' });

    // Debug logging
    console.log('uploadPdfToBackend - File details:', {
      name: pdfFile.name,
      size: pdfFile.size,
      type: pdfFile.type,
      isFile: pdfFile instanceof File,
      isBlob: pdfFile instanceof Blob
    });

    const result = await api.uploads.uploadPDF(pdfFile);

    // Backend returns { filePath: "...", message: "...", ... } directly (lowercase!)
    // Not wrapped in APIResponse format
    const filePath = (result as any).filePath || (result as any).FilePath || result.data?.path || (result as any).path;
    console.log('Upload result:', result);
    console.log('Extracted filePath:', filePath);

    return filePath || null;
  } catch (error) {
    if (error instanceof APIError) {
      const appError = new AppError(`Upload failed: ${error.message}`, {
        category: 'file_upload' as any,
        severity: 'error' as any,
        userMessage: 'Failed to upload PDF to server',
        technicalDetails: { status: error.status, message: error.message }
      });
      handleError(appError, { showToast: false });
      throw appError;
    }
    const networkError = createNetworkError('Failed to upload PDF to backend', error);
    handleError(networkError, { showToast: false });
    throw networkError;
  }
};

export const generatePDF = async (pdfPages, accessPoints, accessPointTypes, mapImageRef) => {
  const { jsPDF } = window.jsPDF;
  const pdf = new jsPDF('portrait', 'mm', 'a4');

  for (let pageIndex = 0; pageIndex < pdfPages.length; pageIndex++) {
    if (pageIndex > 0) {
      pdf.addPage();
    }

    const img = new Image();
    img.src = pdfPages[pageIndex];
    await new Promise(resolve => {
      img.onload = resolve;
    });

    const pdfWidth = pdf.internal.pageSize.getWidth();
    const pdfHeight = pdf.internal.pageSize.getHeight();
    const imgAspect = img.width / img.height;
    const pdfAspect = pdfWidth / pdfHeight;

    let imgWidth, imgHeight;
    if (imgAspect > pdfAspect) {
      imgWidth = pdfWidth;
      imgHeight = pdfWidth / imgAspect;
    } else {
      imgHeight = pdfHeight;
      imgWidth = pdfHeight * imgAspect;
    }

    pdf.addImage(pdfPages[pageIndex], 'PNG', 0, 0, imgWidth, imgHeight);

    const pageAccessPoints = accessPoints.filter(ap => ap.page === pageIndex);
    pageAccessPoints.forEach((ap) => {
      const type = accessPointTypes.find(t => t.id === ap.type);
      if (type) {
        const scaleX = imgWidth / (mapImageRef.current?.naturalWidth || 1);
        const scaleY = imgHeight / (mapImageRef.current?.naturalHeight || 1);

        const pdfX = ap.x * scaleX;
        const pdfY = ap.y * scaleY;

        pdf.setFillColor(type.color);
        pdf.circle(pdfX, pdfY, 3, 'F');

        pdf.setDrawColor(255, 255, 255);
        pdf.setLineWidth(0.5);
        pdf.circle(pdfX, pdfY, 3, 'S');

        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(8);
        pdf.text(ap.name, pdfX + 5, pdfY - 2);
      }
    });
  }

  // Add legend page
  pdf.addPage();
  pdf.setFontSize(16);
  pdf.setTextColor(0, 0, 0);
  pdf.text('Access Points Legend', 20, 20);

  let yPos = 40;
  accessPointTypes.forEach((type) => {
    const typePoints = accessPoints.filter(ap => ap.type === type.id);
    if (typePoints.length > 0) {
      pdf.setFillColor(type.color);
      pdf.circle(25, yPos - 2, 2, 'F');
      pdf.setFontSize(12);
      pdf.text(`${type.name} (${typePoints.length} points)`, 35, yPos);
      yPos += 10;

      typePoints.forEach((point) => {
        pdf.setFontSize(10);
        pdf.setTextColor(100, 100, 100);
        pdf.text(`  • ${point.name} - Page ${point.page + 1}`, 40, yPos);
        yPos += 6;
      });
      yPos += 5;
    }
  });

  pdf.save(`map-with-access-points-${Date.now()}.pdf`);
};

export const calculateVisibleRegion = (viewport, imageDimensions, containerWidth, containerHeight) => {
  const { zoomLevel, panOffset, rotation = 0 } = viewport;

  if (rotation === 0 || rotation === 360) {
    // Top-left corner of visible area (what image point is at screen 0,0)
    const visibleX = -panOffset.x / zoomLevel;
    const visibleY = -panOffset.y / zoomLevel;

    // Bottom-right corner of visible area (what image point is at screen width,height)
    const visibleEndX = (containerWidth - panOffset.x) / zoomLevel;
    const visibleEndY = (containerHeight - panOffset.y) / zoomLevel;

    // Now clamp to ensure we stay within image bounds
    const clampedX = Math.max(0, Math.min(imageDimensions.width, visibleX));
    const clampedY = Math.max(0, Math.min(imageDimensions.height, visibleY));
    const clampedEndX = Math.max(0, Math.min(imageDimensions.width, visibleEndX));
    const clampedEndY = Math.max(0, Math.min(imageDimensions.height, visibleEndY));

    const clampedWidth = clampedEndX - clampedX;
    const clampedHeight = clampedEndY - clampedY;

    return {
      x: clampedX,
      y: clampedY,
      width: clampedWidth,
      height: clampedHeight
    };
  }

  // For rotated images, we need to calculate the bounding box
  // The container shows a rotated view, but we need to find what portion
  // of the ORIGINAL (unrotated) image is visible

  // Calculate dimensions of the rotated image as displayed
  const rotRad = (rotation * Math.PI) / 180;
  const cos = Math.abs(Math.cos(rotRad));
  const sin = Math.abs(Math.sin(rotRad));

  // Rotated image dimensions
  const rotatedWidth = imageDimensions.width * cos + imageDimensions.height * sin;
  const rotatedHeight = imageDimensions.width * sin + imageDimensions.height * cos;

  // Calculate visible area in rotated space
  const visibleRotatedWidth = containerWidth / zoomLevel;
  const visibleRotatedHeight = containerHeight / zoomLevel;

  // Top-left corner of visible area in rotated space
  const visibleRotatedX = -panOffset.x / zoomLevel;
  const visibleRotatedY = -panOffset.y / zoomLevel;

  // Center of the image in rotated space
  const centerRotatedX = rotatedWidth / 2;
  const centerRotatedY = rotatedHeight / 2;

  // Get the four corners of the visible rectangle in rotated space
  const corners = [
    { x: visibleRotatedX, y: visibleRotatedY },
    { x: visibleRotatedX + visibleRotatedWidth, y: visibleRotatedY },
    { x: visibleRotatedX + visibleRotatedWidth, y: visibleRotatedY + visibleRotatedHeight },
    { x: visibleRotatedX, y: visibleRotatedY + visibleRotatedHeight }
  ];

  // Transform corners back to original image space (inverse rotation)
  const centerOrigX = imageDimensions.width / 2;
  const centerOrigY = imageDimensions.height / 2;

  const transformedCorners = corners.map(corner => {
    // Translate to center of rotated space
    const relX = corner.x - centerRotatedX;
    const relY = corner.y - centerRotatedY;

    // Apply inverse rotation
    const invRotRad = -rotRad;
    const cosInv = Math.cos(invRotRad);
    const sinInv = Math.sin(invRotRad);

    const origRelX = relX * cosInv - relY * sinInv;
    const origRelY = relX * sinInv + relY * cosInv;

    // Translate back to original space
    return {
      x: origRelX + centerOrigX,
      y: origRelY + centerOrigY
    };
  });

  // Find bounding box of transformed corners
  const minX = Math.min(...transformedCorners.map(c => c.x));
  const maxX = Math.max(...transformedCorners.map(c => c.x));
  const minY = Math.min(...transformedCorners.map(c => c.y));
  const maxY = Math.max(...transformedCorners.map(c => c.y));

  // Clamp to image bounds
  const clampedX = Math.max(0, Math.min(imageDimensions.width, minX));
  const clampedY = Math.max(0, Math.min(imageDimensions.height, minY));
  const clampedMaxX = Math.max(0, Math.min(imageDimensions.width, maxX));
  const clampedMaxY = Math.max(0, Math.min(imageDimensions.height, maxY));

  return {
    x: clampedX,
    y: clampedY,
    width: clampedMaxX - clampedX,
    height: clampedMaxY - clampedY
  };
};

export const cropImageToViewport = async (imageRef, viewport, imageDimensions, containerRef, dispatch, setPdfFile) => {
  if (!imageRef.current || !containerRef.current) {
    console.error('cropImageToViewport: Missing refs');
    return null;
  }

  const containerRect = containerRef.current.getBoundingClientRect();
  const containerWidth = containerRect.width;
  const containerHeight = containerRect.height;

  // IMPORTANT: Use natural dimensions of the image, not the displayed dimensions
  const naturalWidth = imageRef.current.naturalWidth;
  const naturalHeight = imageRef.current.naturalHeight;

  // Calculate the scale factor between displayed and natural size
  const scaleX = naturalWidth / imageDimensions.width;
  const scaleY = naturalHeight / imageDimensions.height;

  console.log('Crop starting:', {
    viewport,
    displayedDimensions: imageDimensions,
    naturalDimensions: { width: naturalWidth, height: naturalHeight },
    scale: { x: scaleX, y: scaleY },
    containerWidth,
    containerHeight
  });

  // Calculate the visible region using displayed dimensions
  const visibleRegion = calculateVisibleRegion(viewport, imageDimensions, containerWidth, containerHeight);

  // Scale the visible region to natural image coordinates
  const naturalVisibleRegion = {
    x: visibleRegion.x * scaleX,
    y: visibleRegion.y * scaleY,
    width: visibleRegion.width * scaleX,
    height: visibleRegion.height * scaleY
  };

  // Create a temporary image to work with
  const tempImage = new Image();
  tempImage.crossOrigin = 'anonymous';

  return new Promise((resolve) => {
    tempImage.onload = async () => {
      const rotation = viewport.rotation || 0;

      // Step 1: First rotate the ENTIRE original image if needed
      let sourceCanvas;
      let sourceCtx;

      if (rotation !== 0 && rotation !== 360) {
        // Create canvas for rotated full image using NATURAL dimensions
        const rotRad = (rotation * Math.PI) / 180;
        const cos = Math.abs(Math.cos(rotRad));
        const sin = Math.abs(Math.sin(rotRad));

        // Calculate rotated dimensions using natural size
        const rotatedWidth = Math.round(naturalWidth * cos + naturalHeight * sin);
        const rotatedHeight = Math.round(naturalWidth * sin + naturalHeight * cos);

        sourceCanvas = document.createElement('canvas');
        sourceCanvas.width = rotatedWidth;
        sourceCanvas.height = rotatedHeight;
        sourceCtx = sourceCanvas.getContext('2d');

        // Draw rotated image using natural dimensions
        sourceCtx.save();
        sourceCtx.translate(rotatedWidth / 2, rotatedHeight / 2);
        sourceCtx.rotate(rotRad);
        sourceCtx.drawImage(
          tempImage,
          -naturalWidth / 2,
          -naturalHeight / 2,
          naturalWidth,
          naturalHeight
        );
        sourceCtx.restore();
      } else {
        // No rotation needed, use original image with natural dimensions
        sourceCanvas = document.createElement('canvas');
        sourceCanvas.width = naturalWidth;
        sourceCanvas.height = naturalHeight;
        sourceCtx = sourceCanvas.getContext('2d');
        sourceCtx.drawImage(tempImage, 0, 0, naturalWidth, naturalHeight);
      }

      // Step 2: Now crop from the rotated (or original) image using NATURAL dimensions
      const cropCanvas = document.createElement('canvas');
      cropCanvas.width = Math.round(naturalVisibleRegion.width);
      cropCanvas.height = Math.round(naturalVisibleRegion.height);
      const cropCtx = cropCanvas.getContext('2d');

      console.log('Crop canvas size:', { width: cropCanvas.width, height: cropCanvas.height });

      // For rotated images, we need to crop from the rotated canvas
      // For non-rotated, crop from the original position
      if (rotation !== 0 && rotation !== 360) {
        // For rotation, we need to work in the rotated canvas space
        const rotRad = (rotation * Math.PI) / 180;
        const cos = Math.abs(Math.cos(rotRad));
        const sin = Math.abs(Math.sin(rotRad));

        // Use NATURAL dimensions for rotation calculations
        const rotatedWidth = naturalWidth * cos + naturalHeight * sin;
        const rotatedHeight = naturalWidth * sin + naturalHeight * cos;

        // Calculate visible area in rotated space from viewport
        // DON'T clamp to 0 yet - calculate first
        const visibleRotatedX = (-viewport.panOffset.x / viewport.zoomLevel) * scaleX;
        const visibleRotatedY = (-viewport.panOffset.y / viewport.zoomLevel) * scaleY;
        const visibleRotatedEndX = ((containerWidth - viewport.panOffset.x) / viewport.zoomLevel) * scaleX;
        const visibleRotatedEndY = ((containerHeight - viewport.panOffset.y) / viewport.zoomLevel) * scaleY;

        // Now clamp to rotated image bounds
        const clampedX = Math.max(0, Math.min(rotatedWidth, visibleRotatedX));
        const clampedY = Math.max(0, Math.min(rotatedHeight, visibleRotatedY));
        const clampedEndX = Math.max(0, Math.min(rotatedWidth, visibleRotatedEndX));
        const clampedEndY = Math.max(0, Math.min(rotatedHeight, visibleRotatedEndY));

        const clampedWidth = clampedEndX - clampedX;
        const clampedHeight = clampedEndY - clampedY;

        cropCtx.drawImage(
          sourceCanvas,
          Math.round(clampedX),
          Math.round(clampedY),
          Math.round(clampedWidth),
          Math.round(clampedHeight),
          0,
          0,
          cropCanvas.width,
          cropCanvas.height
        );
      } else {
        // Simple crop from original image using NATURAL coordinates
        console.log('Cropping non-rotated image:', naturalVisibleRegion);

        cropCtx.drawImage(
          sourceCanvas,
          Math.round(naturalVisibleRegion.x),
          Math.round(naturalVisibleRegion.y),
          Math.round(naturalVisibleRegion.width),
          Math.round(naturalVisibleRegion.height),
          0,
          0,
          cropCanvas.width,
          cropCanvas.height
        );
      }

      // Step 3: Generate PDF from cropped result
      const mmWidth = cropCanvas.width * 0.264583; // pixels to mm
      const mmHeight = cropCanvas.height * 0.264583;

      const orientation = cropCanvas.width > cropCanvas.height ? 'landscape' : 'portrait';
      const {jsPDF} = window.jspdf;
      const pdf = new jsPDF(orientation, 'mm', [mmWidth, mmHeight]);

      let quality = 0.95;
      let croppedDataUrl;
      let pdfBlob;

      do {
        croppedDataUrl = cropCanvas.toDataURL('image/jpeg', quality);
        pdf.addImage(croppedDataUrl, 'JPEG', 0, 0, mmWidth, mmHeight);

        pdfBlob = pdf.output('blob');
        const sizeMB = pdfBlob.size / (1024 * 1024);

        if (sizeMB > 3 && quality > 0.3) {
          // Too large, reduce quality and try again
          pdf.deletePage(1);
          quality -= 0.1;
        } else {
          break;
        }
      } while (true);

      const filename = `precision-map-crop-${Date.now()}.pdf`;
      const croppedFile = new File([pdfBlob], filename, { type: 'application/pdf' });

      // Set the cropped PDF file - DO NOT upload yet
      // The file will be uploaded when the project is saved
      setPdfFile(croppedFile);

      resolve({
        dataUrl: croppedDataUrl,
        cropRegion: visibleRegion,
        newDimensions: {width: cropCanvas.width, height: cropCanvas.height}
      });
    };

    tempImage.src = imageRef.current.src;
  });
};
