import React, { useEffect, useState, useCallback, useMemo, useRef } from 'react';
import { m } from 'framer-motion';
import {
  useProjectLoader,
  useScreenManager
} from './hooks';

import { WelcomeScreen, LoadingScreen } from './components/screens';
import {
  SCREEN_TYPES,
  animations
} from './utils';
import { useIsSmallScreen } from '@/features/project-setup/hooks';
import { LoadingAP } from '@/features/project-setup/components/screens';
import AP from '@/app/types/AP';
import getDict from '@/lib/dict';
import { NetworkSetupScreen } from './components/screens/NetworkSetupScreen';
import { useNetworkManager } from './hooks/useNetworkManager';
import { useLoader } from '@/shared/providers';
import { useSetupFlow } from '@/features/project-setup/hooks';
import { SetupFlowProps, NetworkData } from './types/setupFlow';
import {EuiPanel, EuiSpacer, EuiTitle} from "@elastic/eui";

const dict = getDict("de");

/**
 * SetupFlowManager - Manages the initial setup flow for the application
 *
 * Flow: Welcome → Loading Network → Network Setup → Loading AP → Success → Upload
 *
 * @param setCurrentScreen - Callback to update parent component's current screen
 * @param status - Current status of the setup
 * @param savedProjects - Array of previously saved projects
 */
export default function SetupFlowManager({
  setCurrentScreen,
  status,
  savedProjects
}: SetupFlowProps) {
  // Hooks
  const { navigateTo } = useScreenManager();
  const { setProgress, completed, loadProjects } = useProjectLoader();
  const { networkName, setNetworkName, saveNetwork, saved, apiLoading } = useNetworkManager();
  const { currentScreen, setLoading, completeCurrentStep, setCurrentStep } = useLoader();

  // Local state
  const [aps, setAPs] = useState<AP[]>([]);
  const networkErrorSetterRef = useRef<((error: string | null) => void) | null>(null);

  // Reactive screen size hook - updates on resize
  const isSmall = useIsSmallScreen();

  // Setup flow management hook
  const { handleSkipNetworkSetup } = useSetupFlow({
    currentScreen,
    completed,
    saved,
    apiLoading,
    navigateTo,
    completeCurrentStep,
    setCurrentStep,
    setLoading
  });

  // Sync current screen with parent (optimized to only update when screen actually changes)
  const previousScreenRef = useRef(currentScreen);
  useEffect(() => {
    if (previousScreenRef.current !== currentScreen) {
      previousScreenRef.current = currentScreen;
      setCurrentScreen(currentScreen);
    }
  }, [currentScreen, setCurrentScreen]);

  // Handle "Get Started" from welcome screen (memoized with stable dependencies)
  const handleGetStarted = useCallback(() => {
    completeCurrentStep(); // Mark current step (0) as complete BEFORE incrementing
    setCurrentStep(prev => prev + 1); // Then move to next step (1)
    loadProjects();
  }, [setCurrentStep, completeCurrentStep, loadProjects]);

  // Callback to store the error setter function from NetworkSetupScreen
  const handleRegisterErrorSetter = useCallback((setter: (error: string | null) => void) => {
    networkErrorSetterRef.current = setter;
  }, []);

  // Handle network setup save (optimized with React 18 batching)
  const handleSaveNetwork = useCallback(async () => {

    if (!networkName.trim()) {
      if (networkErrorSetterRef.current) {
        networkErrorSetterRef.current('Netzwerkname ist erforderlich');
      }
      setLoading(false);
      return;
    }

    // React 18 automatically batches these updates, but we can use startTransition for better UX
    React.startTransition(() => {
      setLoading(true);
      setProgress(0);
      if (networkErrorSetterRef.current) {
        networkErrorSetterRef.current(null);
      }
    });

    const networkData: NetworkData = {
      name: networkName.trim(),
      type: 'business',
    };

    try {
      await saveNetwork(networkData);
    } catch (error: any) {
      const errorMessage = error?.message || 'Unbekannter Fehler aufgetreten';
      if (networkErrorSetterRef.current) {
        networkErrorSetterRef.current(`Netzwerk konnte nicht erstellt werden: ${errorMessage}`);
      }
      setLoading(false);
    }
  }, [networkName, setLoading, setProgress, saveNetwork]);

  // Handle network setup continuation (removed redundant wrapper)
  const handleNetworkContinue = handleSaveNetwork;

  return (
    <>
      {/* Welcome Screen */}
      {currentScreen === SCREEN_TYPES.WELCOME && (
        <WelcomeScreen
          onNext={handleGetStarted}
          status={status}
          savedProjects={savedProjects}
        />
      )}

      {/* Loading Network Screen */}
      {currentScreen === SCREEN_TYPES.LOADING_NETWORK && (
        <LoadingScreen text={dict.networkLoad} />
      )}

      {/* Loading Access Points Screen */}
      {currentScreen === SCREEN_TYPES.LOADING_AP && (
        <LoadingScreen text={dict.apLoad} />
      )}

      {/* Network Setup Screen */}
      {currentScreen === SCREEN_TYPES.NETWORK_SETUP && (
        <NetworkSetupScreen
          networkName={networkName}
          setNetworkName={setNetworkName}
          onContinue={handleNetworkContinue}
          onSkip={handleSkipNetworkSetup}
          onError={handleRegisterErrorSetter}
        />
      )}

      {/* Success Screen - Access Points Found */}
      {currentScreen === SCREEN_TYPES.SUCCESS && (
        <SuccessScreen aps={aps} setAPs={setAPs} isSmall={isSmall} />
      )}
    </>
  );
}

interface SuccessScreenProps {
  aps: AP[];
  setAPs: React.Dispatch<React.SetStateAction<AP[]>>;
  isSmall: boolean;
}

const SuccessScreen: React.FC<SuccessScreenProps> = React.memo(({ aps, setAPs, isSmall }) => {
  const checkmarkStyle = useMemo(() => ({
    fontSize: isSmall ? '48px' : '64px',
    marginBottom: isSmall ? '20px' : '24px',
    filter: 'drop-shadow(0 4px 12px rgba(220, 38, 38, 0.3))'
  }), [isSmall]);

  const progressBarStyle = useMemo(() => ({
    height: '4px',
    backgroundColor: '#dc2626',
    borderRadius: '2px',
    margin: '0 auto',
    maxWidth: '200px'
  }), []);

  const checkmarkTransition = useMemo(() => ({
    type: "spring" as const,
    stiffness: 200,
    damping: 15,
    delay: 0.2
  }), []);

  return (
    <m.div
      key="success"
      variants={animations.pageVariants}
      initial="initial"
      animate="in"
      exit="out"
    >
      <m.div
        variants={animations.containerVariants}
        initial="hidden"
        animate="visible"
      >
        <EuiPanel css={{margin: '0 12px', textAlign: 'center'}}>
          {/* Animated Checkmark */}
          <m.div
            variants={animations.itemVariants}
            initial={{ scale: 0 }}
            animate={{ scale: 1 }}
            transition={checkmarkTransition}
            style={checkmarkStyle}
          >
            ✅
          </m.div>

          {/* Success Title */}
          <m.div variants={animations.itemVariants}>
            <EuiTitle size="l">
              <h1> {dict.accessPoint} {dict.found} </h1>
            </EuiTitle>
          </m.div>

          <EuiSpacer size="m" />

          {/* Access Points Loading Animation */}
          <LoadingAP aps={aps} setAPs={setAPs} />

          <EuiSpacer size="l" />

          {/* Progress Bar Animation - simplified for better performance */}
          <m.div
            initial={{ width: 0 }}
            animate={{ width: '100%' }}
            transition={{ duration: 2, ease: "easeInOut" }}
            style={progressBarStyle}
          />
        </EuiPanel>
      </m.div>
    </m.div>
  );
});

SuccessScreen.displayName = 'SuccessScreen';
