import React, { useRef } from 'react';
import {
  EuiLoadingSpinner,
  EuiText
} from '@elastic/eui';
import styles from '../../styles/animations.module.css';

import getDict from '@/lib/dict';
import { validatePdfFile } from '@/lib/fileValidation';
import { handleError, createValidationError } from '@/lib/errorHandler';

const dict = getDict("de");

interface FileUploadComponentProps {
  onFileSelect: (files: FileList) => void;
  accept: string;
  apiLoading: boolean;
}

export const FileUploadComponent: React.FC<FileUploadComponentProps> = ({ onFileSelect, accept, apiLoading }) => {
  const fileInputRef = useRef<HTMLInputElement>(null);

  const openFileDialog = () => fileInputRef.current?.click();

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files || files.length === 0) return;

    const file = files[0];

    // Validate PDF file
    const validation = validatePdfFile(file);

    if (!validation.valid) {
      const error = createValidationError(validation.error || 'Invalid file', 'file');
      handleError(error, {
        showToast: true,
        logToConsole: true
      });
      // Reset file input
      if (fileInputRef.current) {
        fileInputRef.current.value = '';
      }
      return;
    }

    // File is valid, pass it to the parent component
    if (onFileSelect) {
      onFileSelect(files);
    }
  };


  return (
    <div style={{ position: 'relative' }}>

      <input
        ref={fileInputRef}
        type="file"
        accept={accept}
        onChange={handleFileChange}
        style={{ display: 'none' }}
      />
      <div
        className={styles.hoverScale}
        onClick={openFileDialog}
        style={{
          border: '2px dashed #d1d5db',
          borderRadius: '12px',
          padding: '40px 20px',
          textAlign: 'center',
          cursor: 'pointer',
          transition: 'all 0.3s ease',
          backgroundColor: 'rgba(255, 255, 255, 0.8)',
        }}
        onMouseEnter={(e) => {
          e.currentTarget.style.borderColor = '#dc2626';
          e.currentTarget.style.backgroundColor = 'rgba(220, 38, 38, 0.05)';
        }}
        onMouseLeave={(e) => {
          e.currentTarget.style.borderColor = '#d1d5db';
          e.currentTarget.style.backgroundColor = 'rgba(255, 255, 255, 0.8)';
        }}
      >
        <div style={{ fontSize: '48px', marginBottom: '16px' }}>📁</div>
        <EuiText size="m" style={{ fontWeight: '600', marginBottom: '8px' }}>
          {dict.clickUpload}
        </EuiText>
        <EuiText size="s" color="subdued">
          {dict.pdf} (Max 50MB)
        </EuiText>
        {apiLoading && <EuiText size="s" color="subdued">
          <EuiLoadingSpinner size="m"/>
        </EuiText>}
      </div>
    </div>
  );
};
